clear all; 
close all;
clc;

%% This file estimates the control function specification of the model

global Switchmergers Switchdummies Start_Switch DIROFF_Switch ...
    INSIDER_switch homebI_switch CFcn_switch

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% (o) Provide Inputs/Parameters                                    %%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Parameters:

% Home directory:
cd '~/home';    

% Parameters:

% Choice of Starting Values
Start_Switch = 1;   % Choose format of starting points
% 1 if start points are from AMPL; NEED .sol files for parameters
% 2 if start points are from MNL; NEED a .csv table with starting points

% Specification Choices:
Switchmergers = 1;  % Turn on/off interaction with Merger environment
Switchdummies = 3;  % = 0: no size dummies interaction; = 3: size dummies on

DIROFF_Switch = 1; % DIROFF switch, indicates whether DIR and OFF are separate
% 1 if DIR and OFF are separate variables
% 2 if DIR and OFF are ONE variable, e.g. DIROFF

INSIDER_switch = 0; % INSIDER_switch
% 0 if no insiders interactions
% 1 if only Managers
% 2 if Managers, Directors and Officers

homebI_switch = 1; % Home Bias Interactions Switch
% 0 is off, 
% 1 is on.

CFcn_switch = 1;
% 0 is off, 
% 1 is on.

%% what variables to take in

atsvar = 'ATS';
dirvar = 'DIR';
mergervar = 'takeover_prem';
ownershipvar = 'InstOwn_perc';
insidervar = 'dum15'; 
CFvar = 'ControlFct_resid';

DataInputs = {atsvar,dirvar,mergervar,ownershipvar,insidervar,CFvar};

%% Input Combinations to generate tables in the paper:

% Table 7, column (1)
% Switchmergers = 1;  INSIDER_switch = 0; atsvar = 'ATS'; dirvar = 'DIR';
% Table 7, column (2)
% Switchmergers = 1;  INSIDER_switch = 0; atsvar = 'PillStat'; dirvar = 'DIR';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  (i) load the data (using the script PrepData.m) and generates a .dat file %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

global XCMATu XPMAT T PInc XXP

% call data function
[XCMATu, XPMAT, med, big, Instr] = PrepareDataCF(DataInputs);

%% Declare Globals derived from data or initialized

global NCS NP NT NALT Pvarn Cvarn WW Ni_CF Int Int2 Int3 Int3_1 HomebInt NhomebInt

% parameters
NP = max(XCMATu(:,1)); % number of firms
NT = max(XCMATu(:,2)); % number of years
NALT = max(XPMAT(:,2)); % number of states 
        
NCS = size(XCMATu,1); % number of choice situations (number of rows) 

% numbers of Prod-specific and Cons-specific variables.

Pvarn = size(XPMAT,2) - 2; % excluding stateid timeid
Cvarn = size(XCMATu,2) - 4; % excluding firmid timeid inc head 

% input structure of interactions of the model
% E.g. Int = [1 1;
%       1 1;
%       1 1];  % indicates which of the cons-specific variables will be interacted
             % this matrix has dimensions Pvarn x Cvarn, and element (i,j) indicates whether
             % p-variable i is interacted with c-variable j    
             
Int = [ones(Pvarn,Cvarn-1)];
    
Int2 = sum(Int,1);
Int3 = sum(Int2);
Int3_1 = Pvarn*Cvarn;

HomebInt = [0;1;1;0];
NhomebInt = sum(HomebInt);
NInt = sum(sum(Int));

%%
% create vector NCS of indices that for each choice situation i,t whether t is in T1(i)
% that is, whether i cannot be subject to inertia at time t.

T1 = zeros(NCS,1);
% there is no previous year

T2 = zeros(NCS,1);
% indicates when i could be subject to inertia but switches

T3 = zeros(NCS,1);
% indicates when i could be subject to inertia and does not switch

T1(1) = 1; 
T2(1) = 0; 
T3(1) = 0;  

for x = 2:NCS
    T2(x) = (XCMATu(x,2) > 1)*( XCMATu(x,2) == XCMATu(x-1,2) + 1 )*( XCMATu(x,1) == XCMATu(x-1,1))*( XCMATu(x,3) ~= XCMATu(x-1,3)); 
    % year is greater than 1 & in row before year is -1 & in row before same firm id & in year before different inc
    T3(x) = (XCMATu(x,2) > 1)*( XCMATu(x,2) == XCMATu(x-1,2) + 1 )*( XCMATu(x,1) == XCMATu(x-1,1))*( XCMATu(x,3) == XCMATu(x-1,3));
    % same as former line except in year before same inc
    T1(x) = (T2(x)+T3(x) == 0);
end

nT2 = sum(T2); 

T = [T1 T2 T3];

%% Create Matrix of Presence and PastInc for all firms in database

Pres = zeros(NP,NT);    % initialize empty matrix
Incorp = zeros(NP,NT);

for j = 1:NP
    for t = 1:NT
        h = 1;
        while (XCMATu(h,1)~=j || XCMATu(h,2)~=t) && h<NCS
            h = h+1;
        end
        Pres(j,t) = (h < NCS); % fill in with 1 if found firm j at time t
        Incorp(j,t) = (h < NCS)*XCMATu(h,3);   % fill in with incorp code if found j at time t
    end
end

Pres(XCMATu(NCS,1),XCMATu(NCS,2)) = 1;
Incorp(XCMATu(NCS,1),XCMATu(NCS,2)) = XCMATu(NCS,3);

PrevIncorp = zeros(NP,NT);
PrevIncorp(:,2:20) = Incorp(:,1:19);

PInc = [];

for p = 1:NP
    PInc = [PInc; PrevIncorp(p,Pres(p,:)>0)'];
end

PInc(PInc==0) = 1; % set to 1 if no PInc; this does not matter, since if no PInc, it does not enter in likelihood.

% take only NCS in which XCMATu has relevant years
T1=T1((XCMATu(:,2)>=2),:);
T2=T2((XCMATu(:,2)>=2),:);
T3=T3((XCMATu(:,2)>=2),:);
PInc = PInc(XCMATu(:,2)>=2);
med = med(XCMATu(:,2)>=2);
big = big(XCMATu(:,2)>=2);
Instr = Instr(XCMATu(:,2)>=2);

XCMATu = XCMATu((XCMATu(:,2)>=2),:);
NCS = size(XCMATu,1); % number of choice situations
XCMATu(:,2) = XCMATu(:,2)-1*ones(NCS,1);

NP = max(XCMATu(:,1));
NT = 19;
        
NInt = sum(sum(Int));
NCS = size(XCMATu,1); % number of choice situations

XP = reshape(XPMAT(:,3:(2+Pvarn)),NALT,NT,Pvarn);
XXP = reshape(XP,NT*NALT,Pvarn);

% Add objects useful to calculate Std Errors with CF / Two Steps
WW = [ones(size(XCMATu,1),1),XCMATu(:,6:end-1),Instr,T1];
Ni_CF = XCMATu(:,end);


%% Starting points

StartInputs = {Switchmergers,Switchdummies};

if Start_Switch == 2

startMu = 0;
startSigma = 0;
StartingParams = [StartingParams';startMu;startSigma];

elseif Start_Switch == 1

cd '~/StartParams';

betaCStart = csvread('betaC.sol'); % CAREFUL!

betaPStart = csvread('betaP.sol');
betaHbi = csvread('betaHbi.sol');

homebStart = csvread('homeb.sol');
FEStart = csvread('FE.sol');
FEStart = FEStart(2:length(FEStart));

FE_CFcnStart = csvread('FE_CFcn.sol');
FE_CFcnStart = FE_CFcnStart(2:length(FE_CFcnStart));

muStart = csvread('mu.sol');
sigmaStart = csvread('sigma.sol');

StartingParams = [betaPStart;betaCStart;betaHbi;FEStart;FE_CFcnStart;homebStart;muStart;sigmaStart];
    
end
    
startp = sum(T2)/(nT2+sum(T3)); % starting point for parameter pi

startP =StartingParams(1:Pvarn);
startCa =StartingParams(Pvarn+1:Pvarn+Int3);
startC = reshape(startCa',Pvarn,Cvarn-1);
startHbi = StartingParams(Pvarn+Int3+1:Pvarn+Int3+NhomebInt);

startH = StartingParams(Pvarn+Int3+NALT+NALT-1+NhomebInt);
startFE =[0,StartingParams(Pvarn+Int3+1+NhomebInt:Pvarn+Int3+NALT-1+NhomebInt)'];

startFE_CFcn = [0,StartingParams(Pvarn+NALT+Int3+1+NhomebInt:Pvarn+Int3+NALT+NALT-1+NhomebInt)'];

startmu = StartingParams(Pvarn+Int3+NALT+NALT+NhomebInt);
starts = StartingParams(Pvarn+Int3+NALT+1+NALT+NhomebInt);

startHbi2 = zeros(Cvarn-1,1);
startHbi2(HomebInt==1) = startHbi;

%% CREATE .DAT FILE FOR AMPL

cd '~/home';  

Datfilename = 'inatt_hbi.dat';

fid = fopen(Datfilename, 'w');    % this generates the data file
    fprintf (fid, '# Data generated: %s\n\n', datestr(now));  
    fprintf (fid, 'data; \n\n'); 

    % 0. parameters
    fprintAmplParam(fid, 'NP', NP, 1); % number of DM
    fprintAmplParam(fid, 'NT', NT, 1); % number of choice periods
    fprintAmplParam(fid, 'NALT', NALT, 1); % number of alternatives for each choice (fixed)
    fprintAmplParam(fid, 'NCS', NCS, 1); % number of choice situations
    fprintAmplParam(fid, 'Pvarn', Pvarn, 1); % number of p-level char
    fprintAmplParam(fid, 'Cvarn', Cvarn-1, 1); % number of c-level char
    fprintAmplParam(fid, 'NInt', NInt, 1); % number of interaction variables

    % 1. choices
    fprintAmplParam(fid, 'DataChoice', XCMATu(:,3), 1); % NCSx1 vector of choices    
    fprintAmplParam(fid, 'PInc', PInc, 1); % NCSx1 vector of choices    

    % 1.1 Time indices of every choice instance
    fprintAmplParam(fid, 'TT', XCMATu(:,2), 1); % NCSx1 vector of time indices of choice situations
    
    % 2. covariates
    fprintAmplParam(fid, 'XC', XCMATu(:,5:(3+Cvarn)), 1); % (NTxNP)xCvarn matrix of c-level char
    fprintAmplParam(fid, 'XP', XP, 1); % NT*NALT*Pvarn matrix of p-level char
    fprintAmplParam(fid, 'CFcn', XCMATu(:,end), 1); % (NTxNP)xCvarn matrix of c-level char

    % 2.1 home alternative for each DM/period
    fprintAmplParam(fid, 'CHalt', XCMATu(:,4), 1); % (NTxNP)x1 vector of consumer/time home alternative
    
    % 3. Interactions pattern
    fprintAmplParam(fid, 'Int', Int, 1); % PvarnxCvarn matrix describing interaction pattern
    
    % 3.1 Home Bias Interactions
    fprintAmplParam(fid, 'HomebInt', HomebInt, 1); % PvarnxCvarn matrix describing interaction pattern
    
    % 4. Time/choice switching indices
    fprintAmplParam(fid, 'T1', T1, 1); % NCS vector of period indicators t1
    fprintAmplParam(fid, 'T2', T2, 1); % NCS vector of period indicators t2
    fprintAmplParam(fid, 'T3', T3, 1); % NCS vector of period indicators t3
    
    fprintAmplParam(fid, 'nT2', nT2, 1); % number of ones in T2

    % 5. Starting points
    fprintAmplParam(fid, 'startP', startP, 1); % vector of starting points for betaP
    fprintAmplParam(fid, 'startC', startC, 1); % vector of starting points for betaC
    fprintAmplParam(fid, 'startHbi', startHbi2, 1); % vector of starting points for betaC
    fprintAmplParam(fid, 'startH', startH, 1); % vector of starting points for betaC
    fprintAmplParam(fid, 'startp', startp, 1); % starting point for p
    fprintAmplParam(fid, 'startFE', startFE, 1); % vector of starting points for FE
    fprintAmplParam(fid, 'startFE_CFcn', startFE_CFcn, 1); % vector of starting points for FE
    fprintAmplParam(fid, 'startmu', startmu, 1); % vector of starting points for mu
    fprintAmplParam(fid, 'starts', starts, 1); % vector of starting points for sigma

fclose(fid);    

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% (ii) estimate the model calling AMPL/Knitro                  %%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

!(ampl inatt_hbi_1_DCFcnM.run) > inatt_hbi_1_DCFcnM.out
!(knitroampl LogLikMax -AMPL wantsol=1) > knitroinatt_inatt.out
!(ampl inatt_hbi_2_DCFcnM.run) > inatt_hbi_2_DCFcnM.out

%% collect parameters

betaCEst = csvread('betaC.sol'); 
betaHbiEst = csvread('betaHbi.sol'); 

% the order of the coefficients of interactions is the order of the
% elements of the matrix Int when read column by column.
betaPEst = csvread('betaP.sol');
homebEst = csvread('homeb.sol');
FEEst = csvread('FE.sol');
FEEst = FEEst(2:length(FEEst));

muEst = csvread('mu.sol');
sigmaEst = csvread('sigma.sol');

FE_CFcn_Est = csvread('FE_CFcn.sol');
FE_CFcn_Est = FE_CFcn_Est(2:length(FE_CFcn_Est));

ParamEst = [betaPEst;betaCEst;betaHbiEst;FE_CFcn_Est;FEEst;homebEst;muEst;sigmaEst];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% (iii) compute standard errors                                %%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

global CFcn

Cvarn =  Cvarn-1; 
CFcn= XCMATu(:,end); 
XCMATu = XCMATu(:,1:end-1);

options3 = optimoptions(@fminunc,'MaxIter',1,'MaxFunEvals',1, 'FinDiffType', 'central',...
    'Algorithm','quasi-newton','GradObj','on','DerivativeCheck','off');
[ParamEst2,fval2,exitflag2,output2,grad2,hes] = fminunc(@LogLikG_mh_Cf,ParamEst,options3);

OGrad = OuterGradh_Cf(ParamEst);
BHHH = zeros(length(ParamEst));
for i = 1:NCS
    BHHH =  OGrad(:,i)*OGrad(:,i)' + BHHH;
end

%% Control Fcn standard error correction

OgradAlpha = OuterGradh_Alphas_Cf(ParamEst);

V1_rob = (NCS)*inv(BHHH);
V2 = sum(Ni_CF.^2)/(NCS-size(WW,2))*(WW'*WW)^-1;
diag(sqrt(V2));

A = (1/NCS)*(WW'*WW);
B = (1/NCS)*OGrad*OgradAlpha';
OMgh = (1/NCS)*OGrad*(repmat(Ni_CF,1,size(WW,2)).*WW); 
OMhg = OMgh';

V_rob2 = V1_rob - V1_rob*B*inv(A)*OMhg*V1_rob - V1_rob*OMgh*inv(A)'*B'*V1_rob' + V1_rob*B*(V2*NCS)*B'*V1_rob';
se_CfCorr_rob2 = diag(sqrt(V_rob2/NCS));

csvwrite('se.sol',se_CfCorr_rob2);

t_CfCorr_rob2 = ParamEst ./ se_CfCorr_rob2;
csvwrite('t.sol',t_CfCorr_rob2);
