%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figures_2019.m: runs a collection of models and builds the figures from their output
%
% Sections:
%
% DEEP PARAMETERS & STEADY STATE TARGETS
% MODEL 1a: Flexible-price model (imperfect insurance & inertial wage)
% MODEL 1b: Flexible-price model (imperfect insurance & constant wage)
% MODEL 1c: Flexible-price model (Imperfect insurance & basic Nash wage)
% MODEL 1d. Flexible-price model (perfect insurance & inertial wage)
%   (calls Flexprice.mod)
% MODEL 2a: Policy-rule model (imperfect insurance & inertial wage wage)
% MODEL 2b: Policy-rule model (perfect insurance & inertial wage)
%   (calls Taylor.mod)
% MODEL 3: Baseline imperfect-insurance model (inertial wage & undistorted steady state)
% MODEL 3a: Model 3 without the production subsidy
% MODEL 3b: Model 3 without the wage subsidy
% MODEL 3c: Model 3 without any subsidy
% MODEL 3d: Model 3 without wage inertia
%   (calls Ramsey.mod)
% MODEL 4: Constant-wage model (constant wage & undistorted steady state)
% MODEL 4a: Model 4 without the production subsidy
% MODEL 4b: Model 4 without the wage subsidy
% MODEL 4c: Model 4 without any subsidy
%   (calls Ramsey.mod)
% MODEL 5: Perfect-insurance limit (inertial wage & undistorted steady state)
% MODEL 5a: Model 5 without the production subsidy
% MODEL 5b: Model 5 without the wage subsidy
% MODEL 5c: Model 5 without any subsidy
%   (calls Ramsey.mod)
% MODEL 6: Efficient-wage model (imperfect insurance & inertial wage)
%   (calls Efficient_wage.mod)
% MODEL 7: 
% FIGURE 1: Responses to aggregate shocks under flexible prices
% FIGURE 2: Responses to aggregate shocks under the simple policy rule
% FIGURE 3: Responses to a contractionary productivity shock
% FIGURE 4: Responses to a contractionary cost-push shock
% FIGURE 5: Nominal, real and natural interest rates compared
% FIGURE 6: Responses to a contractionary productivity shock (inertial vs. efficient wage)
% FIGURE 7: Responses to a contractionary cost-push shock (inertial vs. efficient wage)
% FIGURE 8: Responses to a contractionary productivity shock (distorted steady state)
% FIGURE 9: Responses to a contractionary cost-push shock (distorted steady state)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% DEEP PARAMETERS & STEADY STATE TARGETS
% Clearing
clear variables;
close all;
% Employed workers & firm owners
iSS   = 0.005;                                                             % Steady state interest rate
rat   = 0.900;                                                             % Replacement ratio under imperfect insurance
ratlim= 0.993;                                                             % Replacement ratio in perfect-insurance limit
sig   = 0.283;                                                             % Default utility curvatuve for firm owners (s.t. dln(w)/dln(z)=1/3 in efficient-wage model)
ph    = 0.948;                                                             % Default wage inertia (s.t. dln(w)/dln(z)=1/3 in  flex-price model)
varpi = 0.500;                                                               % Firm owners' home production
% Labor market
xw    = 0.045;                                                             % vacancy posting cost over real wage
gamma = 2/3;                                                               % Elasticity of matching function
omega = 0.75;                                                              % fraction of firms not changing prices
fSS   = 0.80;                                                               % Job finding rate
lambdaSS= 0.7;                                                            % Vacancy filling rate
sSS   = 0.05 ;                                                             % Job separation rate
m     = fSS^gamma*(lambdaSS^(1-gamma));                                    % Scaling of the matching function
rho   = sSS/(1-fSS);                                                       % Beginning-of-period job destruction rate 
nSS   = fSS/(sSS+fSS);                                                     % Employment
vSS   = (rho*nSS/(1-nSS*(1-rho))^gamma)^(1/(1-gamma));                     % Vacancies
% Wholesale firms
theta = 6;                                                                 % Elasticity of substitution across wholesale goods
% Shocks
mu_z    = 0.95;                                                            % Persistence of productivity shocks
mu_zeta = 0.95;                                                            % Persistence of cost-push shocks
mu_pol  = 0.85;                                                            % Interest-rule inertia
stdepsz = 0.01;                                                            % Size of labor productivity shock
stdezeta= 0.01/0.9797;                                                     % Size of cost-push shock (1% of baseline steady-state wage)
stdepol = 0.0014;                                                          % Size of monetary policy shock (generates +25 bp annualised)
% Figure specifications
L=30;                                                                      % Lengths of IRFs
X=0:1:L;                                                                   % x axis
y_lim_i = [-30,20];                                                        % ylim for i
y_lim_pi= [-1.5,1.5];                                                      % ylim for pi
y_lim_w = [-1,0.1];                                                        % ylim for w
y_lim_n = [-1.1,0];                                                        % ylim for n
y_lim_Y = [-2,0];                                                          % ylim for Y
LinWi =1.5;                                                                % Line width
MarSi = 3;                                                                 % Marker size

%% MODEL 1a: Flexible-price model (imperfect insurance & inertial wage)
% Parameters
sigma_tilde = sig;                                                         % Risk aversion of firm owners
ratio = rat;                                                               % Replacement ratio                                                               
phi = ph;                                                                  % Wage inertia
tauW = 1/theta;                                                            % Production subsidy
% Steady state
Psi_tilde = (sSS+(1/ratio-1)^-1)^-1;                                       % Precautionary saving coefficient
beta = ((1+iSS)*(1+sSS*(1/ratio-1)))^-1;                                   % Subjective discount factor
gSS  = (theta-1)/(theta*(1-tauW));                                         % Wholesale prices
KSS  = gSS*nSS/(1-omega*beta);                                             % Price setting parameter (numerator)
FSS  = nSS/(1-omega*beta);                                                 % Price setting parameter (denominator)
wSS  = (1+log(ratio)+(xw*fSS^(gamma/(1-gamma))/m^(1/(1-gamma)))*...
    ((1-beta*(1-rho)*(1-gamma*fSS))/(1-gamma)))^-1;                        % Real wage
eta  = xw*wSS;                                                             % Vacancy cost
T    = -wSS*log(ratio);                                                    % Hiring subsidy
tauI = 1- ((1-gamma)*(1-beta*(1-rho)))/(1-beta*(1-rho)*(1-gamma*fSS));     % Corporate tax
CFSS = nSS*(1-wSS) - eta*vSS + varpi;                                      % Total consumption of firm owners
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));                                   % Worker surplus
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));                          % Job value
% Launch Dynare code and save output
save param ratio beta wSS SwSS JSS sigma_tilde gamma eta tauI...
    T tauW phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 1a: Flexible-price model (imperfect insurance & inertial wage)')
dynare flexprice.mod noclearall nolog
z_ALL_z = z_epsz*100;                                                      % Productivity, converted to proportional deviations from steady state (%)
w_IIFPVW_z = w_epsz*100/wSS;                                               % Real wage, converted to proportional deviations from steady state (%)           
n_IIFPVW_z = n_epsz*100/nSS;                                               % Employment, converted to proportional deviations from steady state (%)
R_IIFPVW_z=(w_IIFPVW_z(2:end)-...
    w_IIFPVW_z(1:end-1)-Psi_tilde*s_epsz(2:end)*100).*400;                 % Real rate, converted to annualised basis points (bp)
zeta_ALL_zeta = zeta_ezeta*100;                                            % Cost-push shock, converted to level deviations from steady state (%)
w_IIFPVW_zeta = w_ezeta*100/wSS;                                           % Real wage, converted to proportional deviations from steady state (%)
n_IIFPVW_zeta = n_ezeta*100/nSS;                                           % Employment, converted to proportional deviations from steady state (%)
R_IIFPVW_zeta = (w_IIFPVW_zeta(2:end)-...
    w_IIFPVW_zeta(1:end-1)-Psi_tilde*s_ezeta(2:end)*100).*400;             % Real rate, converted to annualised basis points (bp)
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 1b: Flexible-price model (imperfect insurance & constant wage)
% Parameters
phi = 1;                                                                   % Wage inertia
sigma_tilda = 0;                                                           % So that phi=1 corresponds to the constrained-efficent wage
% Launch Dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 1b: Flexible-price model (imperfect insurance & constant wage)')
dynare flexprice.mod noclearall nolog
w_IIFPCW_z = w_epsz*100/wSS;
n_IIFPCW_z = n_epsz*100/nSS;                        
R_IIFPCW_z = (w_IIFPCW_z(2:end)-w_IIFPCW_z(1:end-1)-Psi_tilde*s_epsz(2:end)*100).*400;
w_IIFPCW_zeta = w_ezeta*100/wSS;
n_IIFPCW_zeta = n_ezeta*100/nSS;
R_IIFPCW_zeta = (w_IIFPCW_zeta(2:end)-w_IIFPCW_zeta(1:end-1)-Psi_tilde*s_ezeta(2:end)*100).*400;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end
%% MODEL 1c: Flexible-price model (imperfect insurance & basic Nash wage)
% Parameters
phi = 0;
sigma_tilde = 1.35;                                                        % so that instant wage response to productivity shock be efficient when phi=0
% Launch Dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 1c: Flexible-price model (imperfect insurance & basic Nash wage)')
dynare flexprice.mod noclearall nolog
w_IIFPVWc_z = w_epsz*100/wSS;
n_IIFPVWc_z = n_epsz*100/nSS;                        
R_IIFPVWc_z = (w_IIFPVWc_z(2:end)-w_IIFPVWc_z(1:end-1)-Psi_tilde*s_epsz(2:end)*100).*400;
w_IIFPVWc_zeta = w_ezeta*100/wSS;
n_IIFPVWc_zeta = n_ezeta*100/nSS;
R_IIFPVWc_zeta = (w_IIFPVWc_zeta(2:end)-w_IIFPVWc_zeta(1:end-1)-Psi_tilde*s_ezeta(2:end)*100).*400;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end
%% MODEL 1d: Flexible-price model (Perfect insurance & inertial wage)
% Parameters
ratio= ratlim                                                       
phi = ph;
sigma_tilde = sig;
% Steady state
Psi_tilde = (sSS+(1/ratio-1)^-1)^-1;
beta = ((1+iSS)*(1+sSS*(1/ratio-1)))^-1;                                   
KSS  = gSS*nSS/(1-omega*beta);                                          
FSS  = nSS/(1-omega*beta);                                                
wSS  = (1+log(ratio)+(xw*fSS^(gamma/(1-gamma))/m^(1/(1-gamma)))*...
    ((1-beta*(1-rho)*(1-gamma*fSS))/(1-gamma)))^-1;                       
eta  = xw*wSS;                                                             
T    = -wSS*log(ratio);                                                    
tauI = 1- ((1-gamma)*(1-beta*(1-rho)))/(1-beta*(1-rho)*(1-gamma*fSS));     
CFSS = nSS*(1-wSS) - eta*vSS + varpi;                                      
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));                                   
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));                          
% Launch Dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 1d: Flexible-price model: Perfect insurance & time-varying wage')
dynare flexprice.mod noclearall nolog
w_PIFPVW_z = w_epsz*100/wSS;            
n_PIFPVW_z = n_epsz*100/nSS;                                
R_PIFPVW_z = (w_PIFPVW_z(2:end)-w_PIFPVW_z(1:end-1)-Psi_tilde*s_epsz(2:end)*100).*400;
w_PIFPVW_zeta = w_ezeta*100/wSS; 
n_PIFPVW_zeta = n_ezeta*100/nSS;
R_PIFPVW_zeta = (w_PIFPVW_zeta(2:end)-w_PIFPVW_zeta(1:end-1)-Psi_tilde*s_ezeta(2:end)*100).*400;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end
% check that firm owners' borrowing constraint always binds
EULEQ_z_cont_PIFPVW   = ((beta + MF_epsz(2:end))).*(1+iSS + (w_epsz(2:end)/wSS-w_epsz(1:end-1)/wSS));
EULEQ_z_exp_PIFPVW    = ((beta - MF_epsz(2:end))).*(1+iSS - (w_epsz(2:end)/wSS-w_epsz(1:end-1)/wSS));
EULEQ_zeta_cont_PIFPVW= ((beta +MF_ezeta(2:end))).*(1+iSS + (w_ezeta(2:end)/wSS-w_ezeta(1:end-1)/wSS));
EULEQ_zeta_exp_PIFPVW = ((beta -MF_ezeta(2:end))).*(1+iSS - (w_ezeta(2:end)/wSS-w_ezeta(1:end-1)/wSS));
if max(EULEQ_z_cont_PIFPVW) > 1
    error('Firm owners borr. constraint not binding after a a contractionary productivity shock')
elseif max(EULEQ_z_exp_PIFPVW) > 1
    error('Firm owners borr. constraint not binding after a a expansionary productivity shock')
elseif max(EULEQ_zeta_cont_PIFPVW) > 1
    error('Firm owners borr. constraint not binding after a a contractionary cost-push shock')
elseif max(EULEQ_zeta_exp_PIFPVW) > 1
    error('Firm owners borr. constraint not binding after a a expansionary cost-push shock')
end
% figure
% plot(EULEQ_z_cont_PIFPVW)
% hold on
% plot(EULEQ_z_exp_PIFPVW)
% plot(EULEQ_zeta_cont_PIFPVW)
% plot(EULEQ_zeta_exp_PIFPVW)
% ylim([0.999 1])
% legend({'Contractionary productivity shock','Expansionary productivity shock','Contractionary cost-push shock','Expansionary cost-push shock'})
% title('Flexible-price model')

%%  MODEL 2a: Policy-rule model (imperfect insurance & inertial wage)
% Parameters
ratio  = rat;
coefpi = 1.5;                                                              % Reaction to inflation
phi    = ph;
% Steady state
beta = ((1+iSS)*(1+sSS*(1/ratio-1)))^-1;                                   
MeSS = beta*(1+sSS*(1/ratio -1));
wSS  = (1+log(ratio)+(xw*fSS^(gamma/(1-gamma))/m^(1/(1-gamma)))*...
    ((1-beta*(1-rho)*(1-gamma*fSS))/(1-gamma)))^-1;                        
eta  = xw*wSS;                                                             
T    = -wSS*log(ratio);                                                    
tauI = 1- ((1-gamma)*(1-beta*(1-rho)))/(1-beta*(1-rho)*(1-gamma*fSS));
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
% Launch Dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI T tauW ...
      phi coefpi mu_z mu_zeta mu_pol stdepsz stdezeta stdepol omega varpi
disp(' ')
disp(' MODEL 2a: Policy-rule model (imperfect insurance & time-varying wage)')
dynare Taylor.mod noclearall nolog
steady
i_IITRVW_z  = i_epsz*40000;                                                % Nominal interest rate, converted to annualised basis points (bp)
pi_IITRVW_z = pi_epsz*400;                                                 % Inflation, converted to annualised percentage points (pp)
n_IITRVW_z  = n_epsz*100/nSS;                                              % Employment, converted to proportional deviations from steady state (%)
w_IITRVW_z  = w_epsz*100/wSS;                                              % Real wage, converted to proportional deviations from steady state (%)
i_IITRVW_zeta = i_ezeta*40000;                                             % Nominal interest rate, converted to annualised basis point (bp)
pi_IITRVW_zeta= pi_ezeta*400;                                              % Inflation, converted to annualised percentage points (pp)
n_IITRVW_zeta = n_ezeta*100/nSS;                                           % Employment, converted to proportional deviations from steady state (%)
pol_ALL_pol = pol_epol*40000;                                              % Nominal interest rate shock, converted to annualised basis points (bp)                                  
i_IITRVW_pol  = i_epol*40000;                                              % Nominal interest rate, converted to annualised basis points (bp)
pi_IITRVW_pol = pi_epol*400;                                               % Inflation, converted to annualised percentage points (pp)
n_IITRVW_pol  = n_epol*100/nSS;                                            % Employment, converted to proportional deviations from steady state (%)
w_IITRVW_pol  = w_epol*100/wSS;                                            % Real wage, converted to proportional deviations from steady state (%)
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
elseif JSS + min(J_epol) <0
    error('Job value turns negative after monetary policy shock')
end

%% MODEL 2b: Policy-rule model (perfect insurance & inertial wage)
% Parameters
phi = ph;                                                          
ratio= ratlim;                                                        
% Steady state
beta = ((1+iSS)*(1+sSS*(1/ratio-1)))^-1;
MeSS = beta*(1+sSS*(1/ratio -1));
KSS  = gSS*nSS/(1-omega*beta);
FSS  = nSS/(1-omega*beta);
wSS  = (1+log(ratio)+(xw*fSS^(gamma/(1-gamma))/m^(1/(1-gamma)))*...
    ((1-beta*(1-rho)*(1-gamma*fSS))/(1-gamma)))^-1;
eta  = xw*wSS; 
T    = -wSS*log(ratio);
tauI = 1- ((1-gamma)*(1-beta*(1-rho)))/(1-beta*(1-rho)*(1-gamma*fSS)); 
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
% Launch Dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI T tauW ...
    phi coefpi mu_z mu_zeta mu_pol stdepsz stdezeta stdepol omega varpi
disp(' ')
disp('MODEL 2b: Policy-rule model (perfect insurance & time-varying wage)')
dynare Taylor.mod noclearall nolog
i_PITRVW_z  = i_epsz*40000;
pi_PITRVW_z = pi_epsz*400;        
n_PITRVW_z  = n_epsz*100/nSS;
w_PITRVW_z  = Y_epsz*100/nSS;
i_PITRVW_zeta = i_ezeta*40000; 
pi_PITRVW_zeta= pi_ezeta*400;
n_PITRVW_zeta = n_ezeta*100/nSS;
i_PITRVW_pol  = i_epol*40000;
pi_PITRVW_pol = pi_epol*400; 
n_PITRVW_pol  = n_epol*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end
% check that firm owner's borrowing constraint always binds
EULEQ_z_cont_PITRVW = ((beta + MF_epsz(2:end))).*(1+iSS+i_epsz(1:end-1))./(1+pi_epsz(2:end));
EULEQ_z_exp_PITRVW    = ((beta - MF_epsz(2:end))).*(1+iSS-i_epsz(1:end-1))./(1-pi_epsz(2:end));
EULEQ_zeta_cont_PITRVW = ((beta + MF_ezeta(2:end))).*(1+iSS+i_ezeta(1:end-1))./(1+pi_ezeta(2:end));
EULEQ_zeta_exp_PITRVW = ((beta - MF_ezeta(2:end))).*(1+iSS-i_ezeta(1:end-1))./(1-pi_ezeta(2:end));
if max(EULEQ_z_cont_PITRVW) > 1
    error('Firm owners borr. constraint not binding after a a contractionary productivity shock')
elseif max(EULEQ_z_exp_PITRVW) > 1
    error('Firm owners borr. constraint not binding after a a expansionary productivity shock')
elseif max(EULEQ_zeta_cont_PITRVW) > 1
    error('Firm owners borr. constraint not binding after a a contractionary cost-push shock')
elseif max(EULEQ_zeta_exp_PITRVW) > 1
    error('Firm owners borr. constraint not binding after a a expansionary cost-push shock')
end
% figure
% plot(EULEQ_z_cont_PITRVW)
% hold on
% plot(EULEQ_z_exp_PITRVW)
% plot(EULEQ_zeta_cont_PITRVW)
% plot(EULEQ_zeta_exp_PITRVW)
% ylim([0.999 1])
% legend({'Contractionary productivity shock','Expansionary productivity shock','Contractionary cost-push shock','Expansionary cost-push shock'})
% title('Policy-rule model')

%% MODEL 3: Baseline imperfect-insurance model (inertial wage & undistorted steady state)
% Parameters
ratio= rat;
sigma_tilde = sig;
tauW = 1/theta;
phi = ph;
% Steady state
beta = ((1+iSS)*(1+sSS*(1/ratio-1)))^-1;
MeSS = beta*(1+sSS*(1/ratio -1));
gSS  = (theta-1)/(theta*(1-tauW)); 
KSS  = gSS*nSS/(1-omega*beta);
FSS  = nSS/(1-omega*beta);
wSS  = (1+log(ratio)+(xw*fSS^(gamma/(1-gamma))/m^(1/(1-gamma)))*...
    ((1-beta*(1-rho)*(1-gamma*fSS))/(1-gamma)))^-1; 
eta  = xw*wSS;
T    = -wSS*log(ratio);
tauI = 1- ((1-gamma)*(1-beta*(1-rho)))/(1-beta*(1-rho)*(1-gamma*fSS));
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
Lambda_tilde = (CFSS^sigma_tilde)/wSS;                                     % Corrected Pareto weight
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
% Launch Dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 3: Baseline imperfect-insurance model (time-varying wage & undistorted steady state)')
dynare Ramsey.mod noclearall nolog
i_IIVW_z = i_epsz*40000; 
pi_IIVW_z = pi_epsz*400; 
w_IIVW_z = w_epsz*100/wSS;            
n_IIVW_z = n_epsz*100/nSS;
Y_IIVW_z = Y_epsz*100/nSS;
zeta_ALL_zeta = zeta_ezeta*100;
i_IIVW_zeta = i_ezeta*40000; 
pi_IIVW_zeta = pi_ezeta*400;
w_IIVW_zeta = w_ezeta*100/wSS; 
n_IIVW_zeta = n_ezeta*100/nSS;
Y_IIVW_zeta = Y_ezeta*100/nSS;
CF_IIVW_z = CF_epsz;
CF_IIVW_zeta = CF_ezeta;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 3a: Model 3 without the production subsidy
% Parameters
tauW = 0;                                                                  
% Steady state
gSS  = (theta-1)/(theta*(1-tauW));
KSS  = gSS*nSS/(1-omega*beta);
wSS  = (gSS+T)*(1+xw*(1-beta*(1-rho))*fSS^(gamma/(1-gamma))...
    /((1-tauI)*m^(1/(1-gamma))) )^-1;
eta  = xw*wSS;
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
Lambda_tilde = (CFSS^sigma_tilde)/wSS;
% Launch dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 3a: Model 3 without the production subsidy')
dynare Ramsey.mod noclearall nolog
i_IIVWa_z = i_epsz*40000;                                                  
pi_IIVWa_z = pi_epsz*400;                                                  
w_IIVWa_z = w_epsz*100/wSS;                                                
n_IIVWa_z = n_epsz*100/nSS;                                                
Y_IIVWa_z = Y_epsz*100/nSS;                                                
i_IIVWa_zeta = i_ezeta*40000;                                              
pi_IIVWa_zeta = pi_ezeta*400;                                              
w_IIVWa_zeta = w_ezeta*100/wSS;                                            
n_IIVWa_zeta = n_ezeta*100/nSS;                                            
Y_IIVWa_zeta = Y_ezeta*100/nSS;                                            
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 3b: Model 3 without the wage subsidy
% Parameters
tauW = 1/theta;                                                            
T = 0;                                                                     
% Steady state
gSS  = (theta-1)/(theta*(1-tauW));
KSS  = gSS*nSS/(1-omega*beta);
wSS  = (gSS+T)*(1+xw*(1-beta*(1-rho))*fSS^(gamma/(1-gamma))...
    /((1-tauI)*m^(1/(1-gamma))) )^-1;
eta  = xw*wSS; 
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
Lambda_tilde = (CFSS^sigma_tilde)/wSS;
% Launch dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde...
    eta tauI T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 3b: Model 3 without the wage subsidy')
dynare Ramsey.mod noclearall nolog
i_IIVWb_z = i_epsz*40000;                                                    
pi_IIVWb_z = pi_epsz*400;                                                    
w_IIVWb_z = w_epsz*100/wSS;                                                
n_IIVWb_z = n_epsz*100/nSS;
Y_IIVWb_z = Y_epsz*100/nSS;
i_IIVWb_zeta = i_ezeta*40000;                                                                          
pi_IIVWb_zeta = pi_ezeta*400;                                                
w_IIVWb_zeta = w_ezeta*100/wSS; 
n_IIVWb_zeta = n_ezeta*100/nSS;
Y_IIVWb_zeta = Y_ezeta*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 3c: Model 3 without any subsidy
% Parameters
tauW = 0;
T = 0;
% Steady state
gSS  = (theta-1)/(theta*(1-tauW));
KSS  = gSS*nSS/(1-omega*beta);
wSS  = (gSS+T)*(1+xw*(1-beta*(1-rho))*fSS^(gamma/(1-gamma))...
    /((1-tauI)*m^(1/(1-gamma))) )^-1;
eta  = xw*wSS;
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
Lambda_tilde = (CFSS^sigma_tilde)/wSS;
% Launch dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde...
    eta tauI T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 3c: Model 3 without any subsidy')
dynare Ramsey.mod noclearall nolog
i_IIVWc_z = i_epsz*40000;
pi_IIVWc_z = pi_epsz*400;
w_IIVWc_z = w_epsz*100/wSS;
n_IIVWc_z = n_epsz*100/nSS;
Y_IIVWc_z = Y_epsz*100/nSS;
i_IIVWc_zeta = i_ezeta*40000;
pi_IIVWc_zeta = pi_ezeta*400;
w_IIVWc_zeta = w_ezeta*100/wSS;
n_IIVWc_zeta = n_ezeta*100/nSS;
Y_IIVWc_zeta = Y_ezeta*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 3d: Model 3 without wage inertia
% Parameters
tauW = 1/theta;                                                            
phi = 0; 
sigma_tilda = 1.35;
% Steady state
gSS  = (theta-1)/(theta*(1-tauW));
KSS  = gSS*nSS/(1-omega*beta);
wSS  = (1+log(ratio)+(xw*fSS^(gamma/(1-gamma))/m^(1/(1-gamma)))*...
    ((1-beta*(1-rho)*(1-gamma*fSS))/(1-gamma)))^-1;
eta  = xw*wSS;  
T    = -wSS*log(ratio);
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
Lambda_tilde = (CFSS^sigma_tilde)/wSS;
% Launch dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 3d: Model 3 without wage inertia')
dynare Ramsey.mod noclearall nolog
i_IIVWd_z = i_epsz*40000;
pi_IIVWd_z = pi_epsz*400; 
w_IIVWd_z = w_epsz*100/wSS;            
n_IIVWd_z = n_epsz*100/nSS;
Y_IIVWd_z = Y_epsz*100/nSS;
i_IIVWd_zeta = i_ezeta*40000; 
pi_IIVWd_zeta = pi_ezeta*400;
w_IIVWd_zeta = w_ezeta*100/wSS;
n_IIVWd_zeta = n_ezeta*100/nSS;
Y_IIVWd_zeta = Y_ezeta*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 4: Constant-wage model (constant wage & undistorted steady state)
% Parameters
phi = 1;
sigma_tilde = 0;
% Steady state
Lambda_tilde = (CFSS^sigma_tilde)/wSS;                                     % Transformed welfare weight
% Save parameters
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
% Launch dynare and save output
disp(' ')
disp('MODEL 4: Constant-wage model (constant wage & undistorted steady state)')
dynare Ramsey.mod noclearall nolog
i_IICW_z = i_epsz*40000; 
pi_IICW_z = pi_epsz*400; 
w_IICW_z = w_epsz*100/wSS;            
n_IICW_z = n_epsz*100/nSS;
Y_IICW_z = Y_epsz*100/nSS;
i_IICW_zeta = i_ezeta*40000; 
pi_IICW_zeta = pi_ezeta*400;
w_IICW_zeta = w_ezeta*100/wSS; 
n_IICW_zeta = n_ezeta*100/nSS;
Y_IICW_zeta = Y_ezeta*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 4a: Model 4 without the production subsidy
% Parameters
tauW = 0;                                                                  
% Steady state
gSS  = (theta-1)/(theta*(1-tauW));
KSS  = gSS*nSS/(1-omega*beta);
wSS  = (gSS+T)*(1+xw*(1-beta*(1-rho))*fSS^(gamma/(1-gamma))...
    /((1-tauI)*m^(1/(1-gamma))) )^-1;
eta  = xw*wSS;
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
Lambda_tilde = (CFSS^sigma_tilde)/wSS;
% Launch dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 4a: Model 4 without the production subsidy')
dynare Ramsey.mod noclearall nolog
i_IICWa_z = i_epsz*40000; 
pi_IICWa_z = pi_epsz*400; 
w_IICWa_z = w_epsz*100/wSS;            
n_IICWa_z = n_epsz*100/nSS;
Y_IICWa_z = Y_epsz*100/nSS;
i_IICWa_zeta = i_ezeta*40000; 
pi_IICWa_zeta = pi_ezeta*400;
w_IICWa_zeta = w_ezeta*100/wSS; 
n_IICWa_zeta = n_ezeta*100/nSS;
Y_IICWa_zeta = Y_ezeta*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 4b: Model 4 without the wage subsidy
% Parameters
tauW = 1/theta;                                                            
T = 0;                                                                     
% Steady state
gSS  = (theta-1)/(theta*(1-tauW));
KSS  = gSS*nSS/(1-omega*beta);
wSS  = (gSS+T)*(1+xw*(1-beta*(1-rho))*fSS^(gamma/(1-gamma))...
    /((1-tauI)*m^(1/(1-gamma))) )^-1;
eta  = xw*wSS;
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
Lambda_tilde = (CFSS^sigma_tilde)/wSS;
% Launch dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 4b: Model 4 without the wage subsidy')
dynare Ramsey.mod noclearall nolog
i_IICWb_z = i_epsz*40000;
pi_IICWb_z = pi_epsz*400; 
w_IICWb_z = w_epsz*100/wSS;            
n_IICWb_z = n_epsz*100/nSS;
Y_IICWb_z = Y_epsz*100/nSS;
i_IICWb_zeta = i_ezeta*40000; 
pi_IICWb_zeta = pi_ezeta*400;
w_IICWb_zeta = w_ezeta*100/wSS; 
n_IICWb_zeta = n_ezeta*100/nSS;
Y_IICWb_zeta = Y_ezeta*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 4c: Model 4 without any subsidy
% Parameters
tauW = 0;                                                                  
T = 0;                                                                     
% Steady state
gSS  = (theta-1)/(theta*(1-tauW));
KSS  = gSS*nSS/(1-omega*beta);
wSS  = (gSS+T)*(1+xw*(1-beta*(1-rho))*fSS^(gamma/(1-gamma))...
    /((1-tauI)*m^(1/(1-gamma))) )^-1;
eta  = xw*wSS;
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
Lambda_tilde = (CFSS^sigma_tilde)/wSS;
% Launch dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 4c: Model 4 without any subsidy')
dynare Ramsey.mod noclearall nolog
i_IICWc_z = i_epsz*40000;
pi_IICWc_z = pi_epsz*400; 
w_IICWc_z = w_epsz*100/wSS;            
n_IICWc_z = n_epsz*100/nSS;
Y_IICWc_z = Y_epsz*100/nSS;
i_IICWc_zeta = i_ezeta*40000; 
pi_IICWc_zeta = pi_ezeta*400;
w_IICWc_zeta = w_ezeta*100/wSS; 
n_IICWc_zeta = n_ezeta*100/nSS;
Y_IICWc_zeta = Y_ezeta*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 5: Perfect-insurance limit (inertial wage & undistorted steady state)
% Parameters
ratio= ratlim;
sigma_tilde = sig;
tauW = 1/theta;
phi = ph;
% Steady state
beta = ((1+iSS)*(1+sSS*(1/ratio-1)))^-1;
MeSS = beta*(1+sSS*(1/ratio -1));
gSS  = (theta-1)/(theta*(1-tauW)); 
KSS  = gSS*nSS/(1-omega*beta);
FSS  = nSS/(1-omega*beta);
wSS  = (1+log(ratio)+(xw*fSS^(gamma/(1-gamma))/m^(1/(1-gamma)))*...
    ((1-beta*(1-rho)*(1-gamma*fSS))/(1-gamma)))^-1;
eta  = xw*wSS; 
T    = -wSS*log(ratio);
tauI = 1- ((1-gamma)*(1-beta*(1-rho)))/(1-beta*(1-rho)*(1-gamma*fSS));
CFSS = nSS*(1-wSS) - eta*vSS + varpi; 
Lambda_tilde = (CFSS^sigma_tilde)/wSS;
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
% Launch Dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde...
    eta tauI T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 5: Perfect-insurance limit (time-varying wage & undistorted steady state)')
dynare Ramsey.mod noclearall nolog
i_PIVW_z = i_epsz*40000; 
pi_PIVW_z = pi_epsz*400; 
w_PIVW_z = w_epsz*100/wSS;            
n_PIVW_z = n_epsz*100/nSS;
Y_PIVW_z = Y_epsz*100/nSS;
i_PIVW_zeta = i_ezeta*40000; 
pi_PIVW_zeta = pi_ezeta*400;
w_PIVW_zeta = w_ezeta*100/wSS; 
n_PIVW_zeta = n_ezeta*100/nSS;
Y_PIVW_zeta = Y_ezeta*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end
% check that firm owners' borrowing constraint binds
EULEQ_z_cont_PIVW    = ((beta + MF_epsz(2:end))).*(1+iSS+i_epsz(1:end-1))./(1+pi_epsz(2:end));
EULEQ_z_exp_PIVW    = ((beta - MF_epsz(2:end))).*(1+iSS-i_epsz(1:end-1))./(1-pi_epsz(2:end));
EULEQ_zeta_cont_PIVW = ((beta + MF_ezeta(2:end))).*(1+iSS+i_ezeta(1:end-1))./(1+pi_ezeta(2:end));
EULEQ_zeta_exp_PIVW = ((beta - MF_ezeta(2:end))).*(1+iSS-i_ezeta(1:end-1))./(1-pi_ezeta(2:end));
if max(EULEQ_z_cont_PIVW) > 1
    error('Firm owners borr. constraint not binding after a a contractionary productivity shock')
elseif max(EULEQ_z_exp_PIVW) > 1
    error('Firm owners borr. constraint not binding after a a expansionary productivity shock')
elseif max(EULEQ_zeta_cont_PIVW) > 1
    error('Firm owners borr. constraint not binding after a a contractionary cost-push shock')
elseif max(EULEQ_zeta_exp_PIVW) > 1
    error('Firm owners borr. constraint not binding after a a expansionary cost-push shock')
end
% figure                   % draw LHS of eq. (25)
% plot(EULEQ_z_cont_PIVW)
% hold on
% plot(EULEQ_z_exp_PIVW)
% plot(EULEQ_zeta_cont_PIVW)
% plot(EULEQ_zeta_exp_PIVW)
% ylim([0.999 1])
% title('Optimal-policy model')
% legend({'Contractionary productivity shock','Expansionary productivity shock','Contractionary cost-push shock','Expansionary cost-push shock'})

%% MODEL 5a: Model 5 without the production subsidy
% Parameters
tauW = 0;
% Steady state
gSS  = (theta-1)/(theta*(1-tauW));
KSS  = gSS*nSS/(1-omega*beta);
FSS  = nSS/(1-omega*beta);
wSS  = (gSS+T)*(1+xw*(1-beta*(1-rho))*fSS^(gamma/(1-gamma))...
    /((1-tauI)*m^(1/(1-gamma))) )^-1;
eta  = xw*wSS;
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
Lambda_tilde = (CFSS^sigma_tilde)/wSS;
% Launch dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 5a: Model 5 without the production subsidy')
dynare Ramsey.mod noclearall nolog
i_PIVWa_z = i_epsz*40000; 
pi_PIVWa_z = pi_epsz*400; 
w_PIVWa_z = w_epsz*100/wSS;            
n_PIVWa_z = n_epsz*100/nSS;
Y_PIVWa_z = Y_epsz*100/nSS;
i_PIVWa_zeta = i_ezeta*40000; 
pi_PIVWa_zeta = pi_ezeta*400;
w_PIVWa_zeta = w_ezeta*100/wSS; 
n_PIVWa_zeta = n_ezeta*100/nSS;
Y_PIVWa_zeta = Y_ezeta*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 5b: Model 5 without the wage subsidy
% Parameters
tauW = 1/theta;                                                            
T = 0;                                                                     
% Steady state
gSS  = (theta-1)/(theta*(1-tauW));
KSS  = gSS*nSS/(1-omega*beta);
FSS  = nSS/(1-omega*beta);
wSS  = (gSS+T)*(1+xw*(1-beta*(1-rho))*fSS^(gamma/(1-gamma))...
    /((1-tauI)*m^(1/(1-gamma))) )^-1;
eta  = xw*wSS; 
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
CFSS = nSS*(1-wSS) - eta*vSS + varpi; 
Lambda_tilde = (CFSS^sigma_tilde)/wSS;
% Launch dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 5b: Model 5 without the wage subsidy')
dynare Ramsey.mod noclearall nolog
i_PIVWb_z = i_epsz*40000;
pi_PIVWb_z = pi_epsz*400; 
w_PIVWb_z = w_epsz*100/wSS;            
n_PIVWb_z = n_epsz*100/nSS;
Y_PIVWb_z = Y_epsz*100/nSS;
i_PIVWb_zeta = i_ezeta*40000; 
pi_PIVWb_zeta = pi_ezeta*400;
w_PIVWb_zeta = w_ezeta*100/wSS; 
n_PIVWb_zeta = n_ezeta*100/nSS;
Y_PIVWb_zeta = Y_ezeta*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 5c: Model 5 without any subsidy
% Parameters
tauW = 0;                                                                  
T = 0;                                                                     
% Steady state
gSS  = (theta-1)/(theta*(1-tauW));
KSS  = gSS*nSS/(1-omega*beta);
FSS  = nSS/(1-omega*beta);
wSS  = (gSS+T)*(1+xw*(1-beta*(1-rho))*fSS^(gamma/(1-gamma))...
    /((1-tauI)*m^(1/(1-gamma))) )^-1;
eta  = xw*wSS;
SwSS = -log(ratio)/(1-beta*(1-sSS-fSS));
JSS  = eta*fSS^(gamma/(1-gamma))/m^(1/(1-gamma));
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
Lambda_tilde = (CFSS^sigma_tilde)/wSS;
% Launch dynare code and save output
save param ratio beta wSS SwSS JSS gamma sigma_tilde eta tauI...
    T tauW Lambda_tilde phi mu_z mu_zeta stdepsz stdezeta omega varpi
disp(' ')
disp('MODEL 5c: Model 5 without any subsidy')
dynare Ramsey.mod noclearall nolog
i_PIVWc_z = i_epsz*40000;
pi_PIVWc_z = pi_epsz*400; 
w_PIVWc_z = w_epsz*100/wSS;            
n_PIVWc_z = n_epsz*100/nSS;
Y_PIVWc_z = Y_epsz*100/nSS;
i_PIVWc_zeta = i_ezeta*40000; 
pi_PIVWc_zeta = pi_ezeta*400;
w_PIVWc_zeta = w_ezeta*100/wSS; 
n_PIVWc_zeta = n_ezeta*100/nSS;
Y_PIVWc_zeta = Y_ezeta*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% MODEL 6: Efficient-wage model
% Parameters
ratio = rat;
tauW = 1/theta;
sigma_tilde = sig;
phi = ph ;
% Steady state  
beta = ((1+iSS)*(1+sSS*(1/ratio-1)))^-1;
MeSS = beta*(1+sSS*(1/ratio -1));
gSS  = (theta-1)/(theta*(1-tauW));
KSS  = gSS*nSS/(1-omega*beta);
FSS  = nSS/(1-omega*beta);
wSS  = (1+log(ratio)+(xw*fSS^(gamma/(1-gamma))/...
    m^(1/(1-gamma)))*((1-beta*(1-rho)*(1-gamma*fSS))/(1-gamma)))^-1; 
tauI = 1-((1-gamma)*(1-beta*(1-rho)))/(1-beta*(1-rho)*(1-gamma*fSS));
T    = -wSS*log(ratio);
eta  = xw*wSS; 
CFSS = nSS*(1-wSS) - eta*vSS + varpi;
Lambda_tilde = (CFSS^sigma_tilde)/wSS;
% Launch dynare and save output
save param ratio beta...                                                
    sigma_tilde varpi...                                                
    m gamma eta rho tauI T fSS sSS  gamma...                            
    theta omega tauW...                                                 
    Lambda_tilde mu_z mu_zeta stdepsz stdezeta omega    
disp(' ')
disp('MODEL 6: Imperfect insurance with constrained-efficient wage')
dynare Efficient_wage.mod noclearall nolog
i_CEW_z = i_epsz*40000;
pi_CEW_z = pi_epsz*400; 
w_CEW_z = w_epsz*100/wSS;            
n_CEW_z = n_epsz*100/nSS;
Y_CEW_z = Y_epsz*100/nSS;
i_CEW_zeta = i_ezeta*40000; 
pi_CEW_zeta = pi_ezeta*400;
w_CEW_zeta = w_ezeta*100/wSS; 
n_CEW_zeta = n_ezeta*100/nSS;
Y_CEW_zeta = Y_ezeta*100/nSS;
% check that wage remains inside the bargaining set
if JSS + min(J_epsz) < 0
   error('Job value turns negative after productivity shock')
elseif JSS + min(J_ezeta) < 0
   error('Job value turns negative after cost-push shock')
end

%% FIGURE 1: Responses to aggregate shocks under flexible prices
figure

subplot(3,4,1)
plot(X,(z_ALL_z(1:L+1)),'k-','LineWidth',LinWi)
hold on
title('Productivity shock')
grid on
xlim([-1 30])
ylim([-1.2 0])
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,2)
plot(X,R_IIFPVW_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,R_PIFPVW_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,R_IIFPVWc_z(1:L+1),'ko','MarkerSize',MarSi)
plot(X,R_IIFPCW_z(1:L+1),'k:','LineWidth',LinWi)
title('Natural interest rate')
grid on
xlim([-1 30])
ylim(y_lim_i)
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

subplot(3,4,3)
plot(X,w_IIFPVW_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,w_PIFPVW_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,w_IIFPVWc_z(1:L+1),'ko','MarkerSize',MarSi)
plot(X,w_IIFPCW_z(1:L+1),'k:','LineWidth',LinWi)
title('Real wage')
grid on
xlim([-1 30])
ylim(y_lim_w)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,4)
plot(X,n_IIFPVW_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PIFPVW_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,n_IIFPVWc_z(1:L+1),'ko','MarkerSize',MarSi)
plot(X,n_IIFPCW_z(1:L+1),'k:','LineWidth',LinWi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,5)
plot(X,(zeta_ALL_zeta(1:L+1)),'k-','LineWidth',LinWi)
title('Cost-push shock')
grid on
xlim([-1 30])
ylim([0 1.2])
xlabel('Quarters','FontSize',8)
ylabel('Level deviations (%)','FontSize',8)

subplot(3,4,6)
plot(X,R_IIFPVW_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,R_PIFPVW_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,R_IIFPVWc_zeta(1:L+1),'ko','MarkerSize',MarSi)
plot(X,R_IIFPCW_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Natural interest rate')
grid on
xlim([-1 30])
ylim(y_lim_i)
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

subplot(3,4,7)
plot(X,w_IIFPVW_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,w_PIFPVW_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,w_IIFPVWc_zeta(1:L+1),'ko','MarkerSize',MarSi)
plot(X,w_IIFPCW_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Real wage')
grid on
xlim([-1 30])
ylim(y_lim_w)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,8)
plot(X,n_IIFPVW_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PIFPVW_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,n_IIFPVWc_zeta(1:L+1),'ko','MarkerSize',MarSi)
plot(X,n_IIFPCW_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

legend({'Imperfect insurance and inertial wage (baseline)','Perfect-insurance limit and inertial wage',...
    'Imperfect insurance and basic Nash wage', 'Imperfect insurance and constant wage'},...
     'Orientation','Horizontal','Position',[0.32 0.24 0.4 0.1],'NumColumns', 2,'FontSize', 10)
legend('boxoff')
print('-bestfit','Figure 1.pdf','-dpdf')

%% FIGURE 2: Responses to aggregate shocks under the simple policy rule
figure

subplot(3,4,1)
plot(X,(pol_ALL_pol(1:L+1)),'k-','LineWidth',LinWi)
title('Mon. policy shock')
grid on
xlim([-1 30])
ylim([-10 60])
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

subplot(3,4,2)
plot(X,i_IITRVW_pol(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,i_PITRVW_pol(1:L+1),'k--','LineWidth',LinWi)
title('Nominal interest rate')
grid on
xlim([-1 30])
ylim([-10 30])
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

subplot(3,4,3)
plot(X,pi_IITRVW_pol(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,pi_PITRVW_pol(1:L+1),'k--','LineWidth',LinWi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(3,4,4)
plot(X,n_IITRVW_pol(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PITRVW_pol(1:L+1),'k--','LineWidth',LinWi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,5)
plot(X,(z_ALL_z(1:L+1)),'k-','LineWidth',LinWi)
title('Productivity shock')
grid on
xlim([-1 30])
ylim([-1.2 0])
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,6)
plot(X,i_IITRVW_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,i_PITRVW_z(1:L+1),'k--','LineWidth',LinWi)
title('Nominal interest rate')
grid on
xlim([-1 30])
ylim(y_lim_i)
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

subplot(3,4,7)
plot(X,pi_IITRVW_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,pi_PITRVW_z(1:L+1),'k--','LineWidth',LinWi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(3,4,8)
plot(X,n_IITRVW_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PITRVW_z(1:L+1),'k--','LineWidth',LinWi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,9)
plot(X,(zeta_ALL_zeta(1:L+1)),'k-','LineWidth',LinWi)
title('Cost-push shock')
grid on
xlim([-1 30])
ylim([0 1.2])
xlabel('Quarters','FontSize',8)
ylabel('Level deviations (%)','FontSize',8)

subplot(3,4,10)
plot(X,i_IITRVW_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,i_PITRVW_zeta(1:L+1),'k--','LineWidth',LinWi)
title('Nominal interest rate')
grid on
xlim([-1 30])
ylim(y_lim_i)
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

subplot(3,4,11)
plot(X,pi_IITRVW_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,pi_PITRVW_zeta(1:L+1),'k--','LineWidth',LinWi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(3,4,12)
plot(X,n_IITRVW_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PITRVW_zeta(1:L+1),'k--','LineWidth',LinWi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

legend({'Imperfect insurance',...
    'Perfect-insurance limit'},'Orientation','horizontal',...
    'Position',[0.3 -0.06 0.4 0.12], 'FontSize', 10)
legend('boxoff')
print('-bestfit','Figure 2.pdf','-dpdf')


%% FIGURE 3: Responses to a contractionary productivity shock
figure

subplot(2,3,1)
plot(X,(z_ALL_z(1:L+1)),'k-','LineWidth',LinWi)
title('Productivity shock')
grid on
xlim([-1 30])
ylim([-1.2 0])
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(2,3,2)
plot(X,i_IIVW_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,i_PIVW_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,i_IICW_z(1:L+1),'k:','LineWidth',LinWi)
title('Nominal interest rate')
grid on
xlim([-1 30])
ylim(y_lim_i)
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

subplot(2,3,3)
plot(X,pi_IIVW_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,pi_PIVW_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,pi_IICW_z(1:L+1),'k:','LineWidth',LinWi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(2,3,4)
plot(X,w_IIVW_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,w_PIVW_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,w_IICW_z(1:L+1),'k:','LineWidth',LinWi)
legend({'Imperfect insurance and inertial wage (baseline)','Perfect-insurance limit',...
    'Constant wage'},'Orientation','horizontal','Position',[0.3 -0.06 0.4 0.12], 'FontSize', 10)
title('Real wage')
grid on
xlim([-1 30])
ylim(y_lim_w)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)
legend('boxoff')

subplot(2,3,5)
plot(X,n_IIVW_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PIVW_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,n_IICW_z(1:L+1),'k:','LineWidth',LinWi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(2,3,6)
plot(X,Y_IIVW_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,Y_PIVW_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,Y_IICW_z(1:L+1),'k:','LineWidth',LinWi)
title('Output')
grid on
xlim([-1 30])
ylim(y_lim_Y)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

print('-bestfit','Figure 3.pdf','-dpdf')

%% FIGURE 4: Responses to a contractionary cost-push shock
figure

subplot(2,3,1)
plot(X,(zeta_ALL_zeta(1:L+1)),'k-','LineWidth',LinWi)
title('Cost-push shock')
grid on
xlim([-1 30])
ylim([0 1.2])
xlabel('Quarters','FontSize',8)
ylabel('Level deviations (%)','FontSize',8)

subplot(2,3,2)
plot(X,i_IIVW_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,i_PIVW_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,i_IICW_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Nominal interest rate')
grid on
xlim([-1 30])
ylim(y_lim_i)
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

subplot(2,3,3)
plot(X,pi_IIVW_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,pi_PIVW_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,pi_IICW_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(2,3,4)
plot(X,w_IIVW_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,w_PIVW_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,w_IICW_zeta(1:L+1),'k:','LineWidth',LinWi)
legend({'Imperfect insurance and inertial wage (baseline)','Perfect-insurance limit', 'Constant wage'},...
    'Orientation','horizontal','Position',[0.3 -0.06 0.4 0.12], 'FontSize', 10)
title('Real wage')
grid on
xlim([-1 30])
ylim(y_lim_w)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)
legend('boxoff')

subplot(2,3,5)
plot(X,n_IIVW_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PIVW_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,n_IICW_zeta(1:L+1),'k:','LineWidth',LinWi)

title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(2,3,6)
plot(X,Y_IIVW_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,Y_PIVW_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,Y_IICW_zeta(1:L+1),'k:','LineWidth',LinWi)

title('Output')
grid on
xlim([-1 30])
ylim(y_lim_Y)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

print('-bestfit','Figure 4.pdf','-dpdf')

%% FIGURE 5: Nominal, real and natural interest rates compared
figure

subplot(2,3,1)
plot(X,(z_ALL_z(1:L+1)),'k-','LineWidth',LinWi)
title('Productivity shock')
grid on
xlim([-1 30])
ylim([-1.2 0])
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(2,3,2)
plot(X,i_PIVW_z(1:L+1),'k--','LineWidth',LinWi)                             
hold on
plot(X,i_PIVW_z(1:L+1)-pi_IIVW_z(2:L+2)*100,'k-','LineWidth',LinWi)
plot(X,R_PIFPVW_z(1:L+1),'ko','MarkerSize',MarSi)
title('Perfect-insurance limit')
grid on
xlim([-1 30])
ylim([0 15])
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

subplot(2,3,3)
plot(X,i_IIVW_z(1:L+1),'k--','LineWidth',LinWi)                             
hold on
plot(X,i_IIVW_z(1:L+1)-pi_IIVW_z(2:L+2)*100,'k-','LineWidth',LinWi)
plot(X,R_IIFPVW_z(1:L+1),'ko','MarkerSize',MarSi)
title('Imperfect insurance')
grid on
xlim([-1 30])
ylim([-15 0])
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

subplot(2,3,4)
plot(X,(zeta_ALL_zeta(1:L+1)),'k-','LineWidth',LinWi)
title('Cost-push shock')
grid on
xlim([-1 30])
ylim([0 1.2])
xlabel('Quarters','FontSize',8)
ylabel('Level deviations (%)','FontSize',8)

subplot(2,3,5)
plot(X,i_PIVW_zeta(1:L+1),'k--','LineWidth',LinWi)                             
hold on
plot(X,i_PIVW_zeta(1:L+1)-pi_IIVW_zeta(2:L+2)*100,'k-','LineWidth',LinWi)
plot(X,R_PIFPVW_zeta(1:L+1),'ko','MarkerSize',MarSi)
title('Perfect-insurance limit')
grid on
xlim([-1 30])
ylim([-2 23])
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)
legend({'Nominal interest rate','Real interest rate', 'Natural interest rate'},...
    'Orientation','horizontal','Position',[0.3 -0.06 0.4 0.12], 'FontSize', 10)
legend('Boxoff')

subplot(2,3,6)
plot(X,i_IIVW_zeta(1:L+1),'k--','LineWidth',LinWi)                             
hold on
plot(X,i_IIVW_zeta(1:L+1)-pi_IIVW_zeta(2:L+2)*100,'k-','LineWidth',LinWi)
plot(X,R_IIFPVW_zeta(1:L+1),'ko','MarkerSize',MarSi)
title('Imperfect insurance')
grid on
xlim([-1 30])
ylim([-18 4])
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

print('-bestfit','Figure 5.pdf','-dpdf')

%% FIGURE 6: Responses to a contractionary productivity shock (inertial vs. efficient wage)
figure

subplot(2,3,1)
plot(X,(z_ALL_z(1:L+1)),'k-','LineWidth',LinWi)
title('Productivity shock')
grid on
xlim([-1 30])
ylim([-1.2 0])
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(2,3,2)
plot(X,i_IIVW_z(1:L+1),'k-','LineWidth',LinWi)  
hold on
plot(X,i_CEW_z(1:L+1),'ko','MarkerSize',MarSi)
title('Nominal interest rate')
grid on
xlim([-1 30])
ylim(y_lim_i)
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

subplot(2,3,3)
plot(X,pi_IIVW_z(1:L+1),'k-','LineWidth',LinWi)  
hold on
plot(X,pi_CEW_z(1:L+1),'ko','MarkerSize',MarSi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(2,3,4)
plot(X,w_IIVW_z(1:L+1),'k-','LineWidth',LinWi)  
hold on
plot(X,w_CEW_z(1:L+1),'ko','MarkerSize',MarSi)
legend({'Inertial wage (baseline)','Efficient wage'},'Orientation','horizontal','Position',[0.3 -0.06 0.4 0.12],'FontSize', 10)
title('Real wage')
grid on
xlim([-1 30])
ylim(y_lim_w)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)
legend('boxoff')

subplot(2,3,5)
plot(X,n_IIVW_z(1:L+1),'k-','LineWidth',LinWi)  
hold on
plot(X,n_CEW_z(1:L+1),'ko','MarkerSize',MarSi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(2,3,6)
plot(X,Y_IIVW_z(1:L+1),'k-','LineWidth',LinWi)  
hold on
plot(X,Y_CEW_z(1:L+1),'ko','MarkerSize',MarSi)
title('Output')
grid on
xlim([-1 30])
ylim(y_lim_Y)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

print('-bestfit','Figure 6.pdf','-dpdf')

%% FIGURE 7: Responses to a contractionary cost-push shock (inertial vs. efficient wage)
figure

subplot(2,3,1)
plot(X,(zeta_ALL_zeta(1:L+1)),'k-','LineWidth',LinWi)
title('Cost-push shock')
grid on
xlim([-1 30])
ylim([0 1.2])
xlabel('Quarters','FontSize',8)
ylabel('Level deviations (%)','FontSize',8)

subplot(2,3,2)
plot(X,i_IIVW_zeta(1:L+1),'k-','LineWidth',LinWi)  
hold on
plot(X,i_CEW_zeta(1:L+1),'ko','MarkerSize',MarSi)
title('Nominal interest rate')
grid on
xlim([-1 30])
ylim(y_lim_i)
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)

subplot(2,3,3)
plot(X,pi_IIVW_zeta(1:L+1),'k-','LineWidth',LinWi)  
hold on
plot(X,pi_CEW_zeta(1:L+1),'ko','MarkerSize',MarSi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(2,3,4)
plot(X,w_IIVW_zeta(1:L+1),'k-','LineWidth',LinWi)  
hold on
plot(X,w_CEW_zeta(1:L+1),'ko','MarkerSize',MarSi)
legend({'Inertial wage (baseline)','Efficient wage'},'Orientation','horizontal','Position',[0.3 -0.06 0.4 0.12],'FontSize', 10)
title('Real wage')
grid on
xlim([-1 30])
ylim(y_lim_w)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)
legend('boxoff')

subplot(2,3,5)
plot(X,n_IIVW_zeta(1:L+1),'k-','LineWidth',LinWi)  
hold on
plot(X,n_CEW_zeta(1:L+1),'ko','MarkerSize',MarSi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(2,3,6)
plot(X,Y_IIVW_zeta(1:L+1),'k-','LineWidth',LinWi)  
hold on
plot(X,Y_CEW_zeta(1:L+1),'ko','MarkerSize',MarSi)
title('Output')
grid on
xlim([-1 30])
ylim(y_lim_Y)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

print('-bestfit','Figure 7.pdf','-dpdf')

%% FIGURE 8: Responses to a contractionary productivity shock (distorted steady state)
figure

subplot(3,4,1)
plot(X,i_IIVWa_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,i_PIVWa_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,i_IICWa_z(1:L+1),'k:','LineWidth',LinWi)
title('Nominal int. rate')
grid on
xlim([-1 30])
ylim([-30,25])
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)
annotation('textbox',[0.2 0.6 0.3 0.3],'String',{'no prod.','subsidy'},'FitBoxToText','on', 'LineStyle','none', 'FontSize',8);

subplot(3,4,2)
plot(X,pi_IIVWa_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,pi_PIVWa_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,pi_IICWa_z(1:L+1),'k:','LineWidth',LinWi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(3,4,3)
plot(X,w_IIVWa_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,w_PIVWa_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,w_IICWa_z(1:L+1),'k:','LineWidth',LinWi)
title('Real wage')
grid on
xlim([-1 30])
ylim(y_lim_w)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,4)
plot(X,n_IIVWa_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PIVWa_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,n_IICWa_z(1:L+1),'k:','LineWidth',LinWi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,5)
plot(X,i_IIVWb_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,i_PIVWb_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,i_IICWb_z(1:L+1),'k:','LineWidth',LinWi)
title('Nominal int. rate')
grid on
xlim([-1 30])
ylim([-30,25])
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)
annotation('textbox',[0.2 0.3 0.3 0.3],'String',{'no wage','subsidy'},'FitBoxToText','on', 'LineStyle','none', 'FontSize',8);

subplot(3,4,6)
plot(X,pi_IIVWb_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,pi_PIVWb_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,pi_IICWb_z(1:L+1),'k:','LineWidth',LinWi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(3,4,7)
plot(X,w_IIVWb_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,w_PIVWb_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,w_IICWb_z(1:L+1),'k:','LineWidth',LinWi)
title('Real wage')
grid on
xlim([-1 30])
ylim(y_lim_w)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,8)
plot(X,n_IIVWb_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PIVWb_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,n_IICWb_z(1:L+1),'k:','LineWidth',LinWi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,9)
plot(X,i_IIVWc_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,i_PIVWc_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,i_IICWc_z(1:L+1),'k:','LineWidth',LinWi)
title('Nominal int. rate')
grid on
xlim([-1 30])
ylim([-30,25])
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)
annotation('textbox',[0.2 0.0 0.3 0.3],'String',{'no subsidy','at all'},'FitBoxToText','on', 'LineStyle','none', 'FontSize',8);

subplot(3,4,10)
plot(X,pi_IIVWc_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,pi_PIVWc_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,pi_IICWc_z(1:L+1),'k:','LineWidth',LinWi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(3,4,11)
plot(X,w_IIVWc_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,w_PIVWc_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,w_IICWc_z(1:L+1),'k:','LineWidth',LinWi)
title('Real wage')
grid on
xlim([-1 30])
ylim(y_lim_w)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,12)
plot(X,n_IIVWc_z(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PIVWc_z(1:L+1),'k--','LineWidth',LinWi)
plot(X,n_IICWc_z(1:L+1),'k:','LineWidth',LinWi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

legend({'Imperfect insurance and inertial wage (baseline)','Perfect-insurance limit',...
    'Constant wage'},'Orientation','horizontal','Position',[0.3 -0.06 0.4 0.12], 'FontSize', 10)
legend('boxoff')
print('-bestfit','Figure 8.pdf','-dpdf')

%% FIGURE 9: Responses to a contractionary cost-push shock (distorted steady state)
figure

subplot(3,4,1)
plot(X,i_IIVWa_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,i_PIVWa_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,i_IICWa_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Nominal int. rate')
grid on
xlim([-1 30])
ylim([-30,25])
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)
annotation('textbox',[0.2 0.6 0.3 0.3],'String',{'no prod.','subsidy'},'FitBoxToText','on', 'LineStyle','none', 'FontSize',8);

subplot(3,4,2)
plot(X,pi_IIVWa_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,pi_PIVWa_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,pi_IICWa_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(3,4,3)
plot(X,w_IIVWa_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,w_PIVWa_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,w_IICWa_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Real wage')
grid on
xlim([-1 30])
ylim(y_lim_w)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,4)
plot(X,n_IIVWa_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PIVWa_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,n_IICWa_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,5)
plot(X,i_IIVWb_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,i_PIVWb_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,i_IICWb_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Nominal int. rate')
grid on
xlim([-1 30])
ylim([-30,25])
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)
annotation('textbox',[0.2 0.3 0.3 0.3],'String',{'no wage','subsidy'},'FitBoxToText','on', 'LineStyle','none', 'FontSize',8);

subplot(3,4,6)
plot(X,pi_IIVWb_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,pi_PIVWb_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,pi_IICWb_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(3,4,7)
plot(X,w_IIVWb_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,w_PIVWb_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,w_IICWb_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Real wage')
grid on
xlim([-1 30])
ylim(y_lim_w)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,8)
plot(X,n_IIVWb_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PIVWb_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,n_IICWb_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,9)
plot(X,i_IIVWc_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,i_PIVWc_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,i_IICWc_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Nominal int. rate')
grid on
xlim([-1 30])
ylim([-30,25])
xlabel('Quarters','FontSize',8)
ylabel('Annual basis points','FontSize',8)
annotation('textbox',[0.2 0.0 0.3 0.3],'String',{'no subsidy','at all'},'FitBoxToText','on', 'LineStyle','none', 'FontSize',8);

subplot(3,4,10)
plot(X,pi_IIVWc_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,pi_PIVWc_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,pi_IICWc_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Inflation')
grid on
xlim([-1 30])
ylim(y_lim_pi)
xlabel('Quarters','FontSize',8)
ylabel('Annual percentage points','FontSize',8)

subplot(3,4,11)
plot(X,w_IIVWc_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,w_PIVWc_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,w_IICWc_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Real wage')
grid on
xlim([-1 30])
ylim(y_lim_w)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

subplot(3,4,12)
plot(X,n_IIVWc_zeta(1:L+1),'k-','LineWidth',LinWi)
hold on
plot(X,n_PIVWc_zeta(1:L+1),'k--','LineWidth',LinWi)
plot(X,n_IICWc_zeta(1:L+1),'k:','LineWidth',LinWi)
title('Employment')
grid on
xlim([-1 30])
ylim(y_lim_n)
xlabel('Quarters','FontSize',8)
ylabel('Proportional deviations (%)','FontSize',8)

legend({'Imperfect insurance and inertial wage (baseline)','Perfect-insurance limit', 'Constant wage'},...
    'Orientation','horizontal','Position',[0.3 -0.06 0.4 0.12], 'FontSize', 10)
legend('boxoff')
print('-bestfit','Figure 9.pdf','-dpdf')







