(* Basic RBC Model, with household's value function and risk aversion.

Eric Swanson, 6/2013
*)

eqns={
  V[t] == C[t]^(1-gamma) /(1-gamma) - chi0 *(L[t]+u[t])^(1+chi) /(1+chi) + beta *Vkp[t],
  Valphaexp[t] == (V[t+1]/VAIMSS)^(1-alpha),
  Vkp[t] == VAIMSS *Valphaexp[t]^(1/(1-alpha)),

(* Output, capital accumulation, and aggregate resource constraint *)
  Y[t] == A[t] *K[t-1]^(1-phi) *L[t]^phi,
  K[t] == (1-delta) *K[t-1] + Y[t] - C[t],

(* Technology process *)
  Log[A[t]] == rho *Log[A[t-1]] + eps[A][t],

(* Euler equation and interest rate *)
  C[t]^-gamma == beta *(1+r[t+1]) *C[t+1]^-gamma *(V[t+1]/Vkp[t])^-alpha,
  r[t] == (1-phi) *Y[t] /K[t-1] - delta,
  chi0 *(L[t] + u[t])^chi == ftheta[t] *beta *(V[t+1]/Vkp[t])^-alpha *(w[t+1] *C[t+1]^-gamma + (1-s-ftheta[t+1])/ftheta[t+1] *chi0 *(L[t+1] + u[t+1])^chi),

(* Labor market *)
  L[t] == (1-s) *L[t-1] + h[t-1],
  h[t] == lmu *u[t]^(1-eta) *v[t]^eta,
  ftheta[t] == lmu *(v[t]/u[t])^eta,
  J[t] == phi *Y[t]/L[t] - w[t] + (1-s) *beta *C[t+1]^-gamma /C[t]^-gamma *(V[t+1]/Vkp[t])^-alpha *J[t+1],
  S[t] == w[t] + (1-s-ftheta[t]) *beta *C[t+1]^-gamma /C[t]^-gamma *(V[t+1]/Vkp[t])^-alpha *S[t+1],
  (1-nu) *S[t] == nu *J[t],
  J[t] == kappa /ftheta[t] *(v[t]/u[t]),

(* equity premium *)
  1/(1+rrf[t]) == beta *C[t+1]^-gamma /C[t]^-gamma *(V[t+1]/Vkp[t])^-alpha,
  p[t] == beta *C[t+1]^-gamma /C[t]^-gamma *(V[t+1]/Vkp[t])^-alpha *(C[t+1]^leverage +p[t+1]),
  ep[t] == (C[t+1]^leverage +p[t+1]) /p[t] - (1+rrf[t])  (* price and eq prem for levered consumption claim *)
(* p[t] == (phi *Y[t]/L[t]-w[t])^leverage + (1-s) *beta *C[t+1]^-gamma /C[t]^-gamma *(V[t+1]/Vkp[t])^-alpha *p[t+1], *)
(* ep[t] == ((phi *Y[t+1]/L[t+1]-w[t+1])^leverage + (1-s)*p[t+1]) /p[t] - (1+rrf[t]) *)  (* price and eq prem for share of firm surplus *)
}

loglinearizevars = {A, C, ftheta, h, J, K, L, p, S, u, v, w, Y}
logrules = Map[#[x_]->E^(#[x])&, loglinearizevars]

parametervals = {
  beta -> .996,
  phi -> .7,
  gamma -> 3,
  chi -> 3,
  alpha -> -130,
  delta -> .00833, (* depreciation rate of 10% per year *)
  s -> .02,
  nu -> .5,
  eta -> .5,
  rho -> .99,
  leverage -> 4,
  chi0 -> beta *Exp[wAIMSS - gamma*CAIMSS] *Exp[fthetaAIMSS]/(s+Exp[fthetaAIMSS]) /0.3^chi,  (* L+u = 0.3 in SS *)
  kappa -> Exp[JAIMSS] *Exp[fthetaAIMSS] /.6, (* set v/u = 0.6 in SS *)
  lmu -> 0.28 *(Exp[vAIMSS-uAIMSS])^-eta (* sets ftheta = 0.28 in SS *)
}

AIMPrecision = 400 ;
AIMZeroTol = 10^-50 ;

(* substitute log transformation rules and parameter values into equations: *)
lsmodel = eqns /.logrules //.SetPrecision[parametervals,AIMPrecision] ;


(* complete variable list:
{A, C, ep, ftheta, h, J, K, L, p, r, rrf, S, u, v, V, Valphaexp, Vkp, w, Y}
*)

SSGuess = {0, -0.1, 0, -1.3, -5.2, 2.1, 3.3, -1.3, 7.6, .004, .004, 2.1, -3.9, -4.4, -229, 1, -229, 0.9, 0.1} ; (* good guess for baseline *)

(* SSGuess = {0, -0.6, 0, -1.3, -5.2, 1.7, 1.8, -1.3, 4.0, .01, .01, 1.7, -3.9, -4.4, -645, 1, -645, 0.5, -0.4} ; good guess for baseline with consumption claim *)



(* find the steady state *)
ss = AIMSS[lsmodel, AIMSSGuess->SSGuess] ;

momSubs = {Sigma->1, mom[A,n_/;OddQ[n]]:>0, mom[A,n_/;EvenQ[n]]:>(n!)/((n/2)!2^(n/2)) *.005^n} ;

(* Compute AIM Bmat, S0Inv: *)
eqnseq0 = Flatten[{lsmodel /.x_Real:>SetPrecision[x,AIMPrecision]}] /.Equal->Subtract ;
allvars = Flatten[Map[Through[AIMVarNames[lsmodel] [t+#]]&, Range[-Max[AIMMaxLag[lsmodel],1], AIMMaxLead[lsmodel]]]] ;
hmat = Chop[Outer[D,eqnseq0,allvars] /.AIMSSSubs /.ss, AIMZeroTol] ;
epsmat = Chop[-Outer[D,eqnseq0,AIMShocks[lsmodel]] /.AIMSSSubs /.ss,AIMZeroTol] ;
{cofb,s0inv} = AIMLinearSoln[hmat,AIMMaxLead[lsmodel]] ;
s0invsmall = N[s0inv .epsmat] ;

(* Compute impulse response functions *)
AIMMatrixPower[x_,n_] := MatrixPower[x,n] ;
AIMMatrixPower[x_,0] := IdentityMatrix[Dimensions[x]] ;
irf = Transpose[Chop[Map[AIMMatrixPower[N[cofb],#] .N[Flatten[s0invsmall]]&, Range[0,180]]]] ;

Kpos = Position[AIMVarNames[lsmodel], K][[1,1]] ;
Ypos = Position[AIMVarNames[lsmodel], Y][[1,1]] ;
wpos = Position[AIMVarNames[lsmodel], w][[1,1]] ;
rpos = Position[AIMVarNames[lsmodel], r][[1,1]] ;
Cpos = Position[AIMVarNames[lsmodel], C][[1,1]] ;
Lpos = Position[AIMVarNames[lsmodel], L][[1,1]] ;
upos = Position[AIMVarNames[lsmodel], u][[1,1]] ;
vpos = Position[AIMVarNames[lsmodel], v][[1,1]] ;
Spos = Position[AIMVarNames[lsmodel], S][[1,1]] ;
(*Zpos = Position[AIMVarNames[lsmodel], Z][[1,1]] ;*)
ppos = Position[AIMVarNames[lsmodel], p][[1,1]] ;
eppos = Position[AIMVarNames[lsmodel], ep][[1,1]] ;


(* Start computing unconditional moments *)
momchop = 10^-7 ;
N[ss]

(* First, define some useful placekeeping variables *)
vars = AIMStateVars[lsmodel] ;
args = AIMGenericArgs[lsmodel] ;
lagvars = AIMLagVars[lsmodel] ;
lagvarpos = Flatten[Map[Position[AIMVarNames[lsmodel],#]&, Map[Head,lagvars]]] ;

xpos = Range[Length[lagvars]] ;
epspos = Range[Length[lagvars]+1,Length[AIMStateVars[lsmodel]]] ;
varsx = ReplacePart[AIMStateVars[lsmodel],Thread[epspos->0]] ;
varseps = ReplacePart[AIMStateVars[lsmodel],Thread[xpos->0]] ;

(* Compute Ex^2 to 2nd order *)
Print["Computing Ex^2 to 2nd order, time is ", Date[]] ;
sigmaeps = s0invsmall .DiagonalMatrix[AIMShocks[lsmodel] /.eps[x_][_]->mom[x,2] /.momSubs] .Transpose[s0invsmall] ;
b = N[cofb] ;
sigmax = FixedPoint[(sum = # + b.#.Transpose[b]; b = b.b; sum)&, sigmaeps, 8] ; (* doubling algorithm *)
subsorder2 = Intersection[Thread[Flatten[Outer[Times,lagvars,lagvars]] -> Flatten[sigmax[[lagvarpos,lagvarpos]]]]] ;


(* Compute Ex to 2nd order *)
Print["Computing Ex to 2nd order, time is ", Date[]] ;
mua = AIMSoln[lsmodel,{},2,AIMZeroTol,AIMPrecision] /.x_Real:>N[x] /.momSubs ;
mucoeffs = CoefficientArrays[mua,args] ;

mub = Expand[Normal[mucoeffs[[3]] .vars.vars]] /.eps[x_][t]^2->mom[x,2] /.eps[_][_]->0 /.momSubs /.subsorder2 ;

b = N[cofb] ;
mu = FixedPoint[(sum = # + b.# ; b = b.b; sum)&, mub, 8] ; (* doubling *)  (* NOTE: need to add AIMSS if you want levels! *)

ppos = Position[AIMVarNames[lsmodel], p][[1,1]] ;
eppos = Position[AIMVarNames[lsmodel], ep][[1,1]] ;

Print["Fixed-labor Risk Aversion R^f:  ", N[gamma + alpha *(1-gamma) //.parametervals/.ss]] ;

Print["Risk Aversion R^c:  ", N[gamma / (1 + gamma/chi *Exp[wAIMSS + LAIMSS - CAIMSS] *(s+Exp[fthetaAIMSS])/(rAIMSS+s+Exp[fthetaAIMSS]))
      + alpha * (1-gamma) /(1 - chi0 *(1-gamma)/(1+chi) *(Exp[LAIMSS]+Exp[uAIMSS])^(1+chi) /Exp[CAIMSS]^(1-gamma)) //.parametervals /.ss]] ;

Print["equity premium:  ", Chop[(AIMSeries[lsmodel,4][[eppos,2]]) *1200 /.momSubs /.Thread[AIMStateVars[lsmodel]->0]]] ;



irf = Transpose[Chop[Map[AIMMatrixPower[N[cofb],#] .N[Flatten[s0invsmall]] *0.5 &, Range[0,120]]]] ; (* 1 std dev shock 0.5 *)

Print[GraphicsGrid[{{ListLinePlot[irf[[Cpos]], AxesLabel->{"","C[t]"}, PlotRange->{{0,120},{0,0.5}},ImageSize->Medium],
                 ListLinePlot[irf[[rpos]], AxesLabel->{"","r[t]"}, PlotRange->{{0,120},{-.02,.02}}],
                 ListLinePlot[irf[[Ypos]], AxesLabel->{"","Y[t]"}, PlotRange->{{0,120},{0,1}}]},
                 {ListLinePlot[irf[[Kpos]], AxesLabel->{"","K[t]"}, PlotRange->{{0,120},{0,.5}}],
                 ListLinePlot[irf[[Lpos]], AxesLabel->{"","L[t]"}, PlotRange->{{0,120},{-0.5,0.5}}],
		 ListLinePlot[irf[[Spos]], AxesLabel->{"","S[t]"}, PlotRange->{{0,120},{0,0.5}}]}}]] ;

(* Correlation between firm surplus and consumption: *)
(* sigmax[[Spos, Cpos]] / Sqrt[sigmax[[Spos, Spos]]] /Sqrt[sigmax[[Cpos, Cpos]]] *)
