(* Basic RBC Model, with household's value function and risk aversion.

Eric Swanson, 6/2013
*)

eqns={
 (* Output, capital accumulation, and aggregate resource constraint *)
  Y[t] == A[t] *K[t-1]^(1-phi) L[t]^phi,
  K[t] == (1-delta) *K[t-1] + Y[t] - C[t],

 (* Technology process *)
  Log[A[t]] == rho *Log[A[t-1]] + eps[A][t],

 (* Euler equation and interest rate *)
  C[t]^-gamma == beta *(1+r[t+1]) *C[t+1]^-gamma,
  r[t] == (1-phi) *Y[t] /K[t-1] - delta,
  chi0 *(L[t] + u[t])^chi == ftheta[t] *beta *(w[t+1] *C[t+1]^-gamma + (1-s-ftheta[t+1])/ftheta[t+1] *chi0 *(L[t+1] + u[t+1])^chi),

 (* Labor market *)
  L[t] == (1-s) *L[t-1] + h[t-1],
  h[t] == lmu *u[t]^(1-eta) *v[t]^eta,
  ftheta[t] == lmu *(v[t]/u[t])^eta,
  J[t] == phi *Y[t]/L[t] - w[t] + (1-s) *beta *C[t+1]^-gamma /C[t]^-gamma *J[t+1],
  S[t] == w[t] + (1-s-ftheta[t]) *beta *C[t+1]^-gamma /C[t]^-gamma *S[t+1],
  (1-nu) *S[t] == nu *J[t],
  J[t] == kappa /ftheta[t] *(v[t]/u[t]),

 (* equity premium *)
  1/(1+rrf[t]) == beta *C[t+1]^-gamma /C[t]^-gamma,
  p[t] == beta *C[t+1]^-gamma /C[t]^-gamma *(C[t+1]^leverage +p[t+1]),  (* price of consumption claim *)
  ep[t] == (C[t+1]^leverage +p[t+1]) /p[t] - (1+rrf[t])
}

loglinearizevars = {A, C, ftheta, h, J, K, L, p, S, u, v, w, Y}
logrules = Map[#[x_]->E^(#[x])&, loglinearizevars]

parametervals = {
  beta -> .996,
  phi -> .7,
  gamma -> 100,
  chi -> 100,
  delta -> .00833,
  s -> .02,
  nu -> .5,
  eta -> .5,
  rho -> .99,
  leverage -> 4,
  chi0 -> beta *Exp[wAIMSS - gamma*CAIMSS] *Exp[fthetaAIMSS]/(s+Exp[fthetaAIMSS]) /0.3^chi,  (* sets L+u = 0.3 in SS *)
  kappa -> Exp[JAIMSS] *Exp[fthetaAIMSS] /.6, (* sets v/u = 0.6 in SS *)
  lmu -> .28 *(Exp[vAIMSS-uAIMSS])^-eta (* sets ftheta = 0.28 in SS *)
}

(* Note: the model above is highly curved (i.e., gamma and chi are large), so AIMPrecision must be set to a large number and
AIMZeroTol to a small number in order to get an accurate numerical solution.  Note that high values of chi and low values of
ftheta need to use especially high values here, like 1200 and 10^-400 *)
AIMPrecision = 400 ;
AIMZeroTol = 10^-70 ;

(* substitute log transformation rules and parameter values into equations: *)
lsmodel = eqns /.logrules //.SetPrecision[parametervals,AIMPrecision] ;


(* complete variable list:
{A, C, ep, ftheta, h, J, K, L, p, r, rrf, S, u, v, w, Y}
*)

SSGuess = {0, -0.15, 0, -1.3, -5.2, 2.1, 3.3, -1.3, 5.0, .004, .004, 2.1, -3.9, -4.4, 0.9, 0.1} ; (* good guess for baseline parameters *)

(* SSGuess = {0, -0.2454, 0, -2.303, -5.299, 2.925, 3.171, -1.387, 4.536, .004, .004, 2.925, -2.996, -3.507, 0.834, -0.019} ; good guess for ftheta=0.1 *)

(* SSGuess = {0, -0.4, 0, -3, -5.5, 3.3, 3.0, -1.5, 3.9, .004, .004, 3.34, -2.45, -3, 0.73, -0.17} ; good guess for ftheta=0.05 *)

(* SSGuess = {0, -0.65, 0, -3.69, -5.705, 3.63, 2.765, -1.79, 2.91, .004, .004, 3.63, -2.015, -2.52, 0.61, -0.425} ; good guess for ftheta=0.025 *)

(* SSGuess = {0, -1.16, 0, -4.605, -6.215, 3.86, 2.25, -2.3, 0.87, .004, .004, 3.86, -1.61, -2.12, 0.475, -0.94} ; good guess for ftheta=0.01 *)

(* SSGuess = {0, -1.60, 0, -5.21, -6.66, 3.94, 1.79, -2.75, -0.95, .004, .004, 3.94, -1.44, -1.95, 0.41, -1.38} ; good guess for ftheta=0.005 *)


(* find the steady state *)
ss = AIMSS[lsmodel, AIMSSGuess->SSGuess] ;

momSubs = {Sigma->1, mom[A,n_/;OddQ[n]]:>0, mom[A,n_/;EvenQ[n]]:>(n!)/((n/2)!2^(n/2)) *.005^n} ;

(* Compute AIM Bmat, S0Inv: *)
eqnseq0 = Flatten[{lsmodel /.x_Real:>SetPrecision[x,AIMPrecision]}] /.Equal->Subtract ;
allvars = Flatten[Map[Through[AIMVarNames[lsmodel] [t+#]]&, Range[-Max[AIMMaxLag[lsmodel],1], AIMMaxLead[lsmodel]]]] ;
hmat = Chop[Outer[D,eqnseq0,allvars] /.AIMSSSubs /.ss, AIMZeroTol] ;
epsmat = Chop[-Outer[D,eqnseq0,AIMShocks[lsmodel]] /.AIMSSSubs /.ss,AIMZeroTol] ;
{cofb,s0inv} = AIMLinearSoln[hmat,AIMMaxLead[lsmodel]] ;
s0invsmall = N[s0inv .epsmat] ;

(* Compute impulse response functions *)
AIMMatrixPower[x_,n_] := MatrixPower[x,n] ;
AIMMatrixPower[x_,0] := IdentityMatrix[Dimensions[x]] ;
irf = Transpose[Chop[Map[AIMMatrixPower[N[cofb],#] .N[Flatten[s0invsmall]]&, Range[0,180]]]] ;

Kpos = Position[AIMVarNames[lsmodel], K][[1,1]] ;
Ypos = Position[AIMVarNames[lsmodel], Y][[1,1]] ;
wpos = Position[AIMVarNames[lsmodel], w][[1,1]] ;
rpos = Position[AIMVarNames[lsmodel], r][[1,1]] ;
Cpos = Position[AIMVarNames[lsmodel], C][[1,1]] ;
Lpos = Position[AIMVarNames[lsmodel], L][[1,1]] ;
upos = Position[AIMVarNames[lsmodel], u][[1,1]] ;
vpos = Position[AIMVarNames[lsmodel], v][[1,1]] ;
ppos = Position[AIMVarNames[lsmodel], p][[1,1]] ;
eppos = Position[AIMVarNames[lsmodel], ep][[1,1]] ;


(* Start computing unconditional moments *)
momchop = 10^-7 ;
N[ss]

(* First, define some useful placekeeping variables *)
vars = AIMStateVars[lsmodel] ;
args = AIMGenericArgs[lsmodel] ;
lagvars = AIMLagVars[lsmodel] ;
lagvarpos = Flatten[Map[Position[AIMVarNames[lsmodel],#]&, Map[Head,lagvars]]] ;

xpos = Range[Length[lagvars]] ;
epspos = Range[Length[lagvars]+1,Length[AIMStateVars[lsmodel]]] ;
varsx = ReplacePart[AIMStateVars[lsmodel],Thread[epspos->0]] ;
varseps = ReplacePart[AIMStateVars[lsmodel],Thread[xpos->0]] ;

(* Compute Ex^2 to 2nd order *)
Print["Computing Ex^2 to 2nd order, time is ", Date[]] ;
sigmaeps = s0invsmall .DiagonalMatrix[AIMShocks[lsmodel] /.eps[x_][_]->mom[x,2] /.momSubs] .Transpose[s0invsmall] ;
b = N[cofb] ;
sigmax = FixedPoint[(sum = # + b.#.Transpose[b]; b = b.b; sum)&, sigmaeps, 8] ; (* doubling algorithm *)
subsorder2 = Intersection[Thread[Flatten[Outer[Times,lagvars,lagvars]] -> Flatten[sigmax[[lagvarpos,lagvarpos]]]]] ;


(* Compute Ex to 2nd order *)
(*
Print["Computing Ex to 2nd order, time is ", Date[]] ;
mua = AIMSoln[lsmodel,{},2,AIMZeroTol,AIMPrecision] /.x_Real:>N[x] /.momSubs ;
mucoeffs = CoefficientArrays[mua,args] ;

mub = Expand[Normal[mucoeffs[[3]] .vars.vars]] /.eps[x_][t]^2->mom[x,2] /.eps[_][_]->0 /.momSubs /.subsorder2 ;
*)

b = N[cofb] ;
mu = FixedPoint[(sum = # + b.# ; b = b.b; sum)&, mub, 8] ; (* doubling *)  (* NOTE: need to add AIMSS if you want levels! *)

ppos = Position[AIMVarNames[lsmodel], p][[1,1]] ;
eppos = Position[AIMVarNames[lsmodel], ep][[1,1]] ;

Print["Risk Aversion R^c:  ", N[gamma / (1 + gamma/chi *Exp[wAIMSS + LAIMSS - CAIMSS] 
                                           *(s+Exp[fthetaAIMSS])/(rAIMSS+s+Exp[fthetaAIMSS])) //.parametervals /.ss]] ;

Print["equity premium:  ", Chop[(AIMSeries[lsmodel,4][[eppos,2]]) *1200 /.momSubs /.Thread[AIMStateVars[lsmodel]->0]]] ;

