% Replication Code for Jobs and Matches: Quits, Replacement Hiring, and Vacancy Chains
% by Yusuf Mercan and Benjamin Schoefer
% July 2019
% For questions contact Yusuf Mercan at yusuf.mercan@unimelb.edu.au

%% Housekeeping
clear
clc
close all

addpath(genpath('./export_fig-master'));

fileID = fopen('param_baseline.txt','w'); fclose(fileID);
fileID = fopen('param_extended.txt','w'); fclose(fileID);
%% Plot options
set(groot,'defaultAxesTickLabelInterpreter','latex'); 
set(groot,'defaultLegendInterpreter','latex');
set(groot,'defaultTextInterpreter','latex');
set(groot,'defaultAxesFontSize', 15)
    
line_alpha = 1;

ls.lineSpec1 = {'lineWidth', 2, 'lineStyle', '-' , 'Color', [rgb('DarkGreen') line_alpha]};
ls.lineSpec2 = {'lineWidth', 2, 'lineStyle', '--', 'Color', [rgb('Navy') line_alpha]};
ls.lineSpec3 = {'lineWidth', 2, 'lineStyle', '-.', 'Color',[ rgb('Maroon') line_alpha]};

ls.lineSpec44 = {'lineWidth', 2, 'lineStyle', '-' , 'Color', rgb('Gray')};
ls.lineSpec4 = {'lineWidth', 2, 'lineStyle', ':' , 'Color',  rgb('Black')};

ls.lineSpec5 = {'lineWidth', 2, 'lineStyle', '-' ,'Marker','o', 'Color', [rgb('DarkGreen') line_alpha]};
ls.lineSpec6 = {'lineWidth', 2, 'lineStyle', '--','Marker','o' , 'Color', [rgb('Navy') line_alpha]};
ls.lineSpec7 = {'lineWidth', 2, 'lineStyle', '-.','Marker','o',  'Color', [rgb('Maroon') line_alpha]};

ls.lineSpec55 = {'lineWidth', 2, 'lineStyle', ':', 'Color', [rgb('DarkGreen') line_alpha]};
ls.lineSpec66 = {'lineWidth', 2, 'lineStyle', ':', 'Color', [rgb('Navy') line_alpha]};

ls.lineSpecAux = {'lineWidth', 2, 'lineStyle', 'none' , 'Color', rgb('White')};

ls.gridLineSpec = {'gridlinestyle','--'};
ls.fontSpec = {'FontSize',15};


% change to the path you want the model figures to be saved
ls.graphPath = '../Output/';

% For the extended model change to '_endogenous', 
% For the baseline model in the appendix change to '_appendix'
ls.figPrefix = ''; 

%% Model Parameters

% **************************** exogenously chosen *************************
par.beta  = 0.9967;  % discount factor
par.rhoZ = 0.975;    % persistence of labor productivity
par.sigmaZ = 0.0044; % standard deviation of labor productivity
par.phi = 0.5;       % worker bargaining share
par.eta = 0.5;       % elasticity of matching function
par.b = 0.9;         % unemployment benefit

par.k1 = 0.1;       % fixed entry cost parameter 1
par.k2 = 0;         % convexity of vacancy creation cost: coefficient on n
par.gamma = 1;      % share of vacancies that are reposted

par.psi = 1;        % elasticity of search cost
par.A   = 1;        % search cost scale parameter

par.xiBad = 0;      % disamenity in bad job 
% *************************************************************************

% **************************** endogenously chosen ************************
par.delta = 0.021;  % obsoletion probability
par.s = 0.018;      % exogenous separation probability
par.kappa = 0.1;    % flow cost of posting vacancy
par.mu = 0.636;     % efficiency of matching function (redundant in CES case)
par.lambda = 1;     % relative efficiency of OJS   

par.pBad  = 0;      % probability of downgrading to bad job: extended model
par.pGood = 0;      % probability of upgrading to good job: extended model
% *************************************************************************

% functional forms
func.k_n = @(n, k1, k2, nSS)  k1 + k2 .* (n-nSS)./nSS; % creation cost of vacancy

func.f_theta = @(theta,eta,mu) mu.*theta.^(1-eta); % worker contact rate
func.q_theta = @(theta,eta,mu) mu.*theta.^-eta; % firm contact rate
%% State space
glob.nXi  = 2; % number of job types

% steady state model
glob.nZ = 1;
glob.zGrid = 1;
glob.Pz = 1;
glob.masterPz   = kron(glob.Pz,eye(glob.nXi));
glob.output     = kron(glob.zGrid,ones(glob.nXi,1));
glob.onesnXi    = ones(1,glob.nXi);
glob.selection  = (1:glob.nXi:(glob.nXi*glob.nZ))';
%% Computational Parameters
comp.tolerVFI = 1e-6;
comp.maxIterVFI = 1000;

comp.tolerDist = 5e-6;
comp.maxIterDist = 1000;

comp.tolerTight = 1e-3;
comp.maxIterTight = 1000;

comp.tolerTrans = 1e-3;
comp.maxIterTrans = 5000;

comp.omega = 0.9;       % damepening parameter in steady state
comp.omegaTrans = 0.97; % dampening parameter in transition

comp.thUB = 10; % upper bound for tightness in bisection
comp.thLB = 0;  %lower bound for tightness in bisection

comp.options = optimset('Display', 'iter');
comp.calibOptions  = optimoptions('fmincon','Display','iter','UseParallel',true);
%% 
calibrate_baseline = 1; % calibrate baseline model
calibrate_extended = 0; % calibrate extended model
transition = 1; % solve transition

%% Calibrate baseline model
if calibrate_baseline == 1
    fprintf('*** Calibrating baseline ***\n')
    targetMom = [0.057;0.56;0.90;0.45;0.025];
    
    par.b = 0.90;
    par.xiBad = 0;
    par.pBad = 0;
    par.pGood = 0;
    
    func.c_s = @(s,A,psi)  zeros(length(s),1); % no search cost
    func.c_sprime_inv = @(c,A,psi) ones(length(c),1); % unit search effort
    obj = @(endopar) calibrationObjective(endopar,targetMom,par,func,glob,comp);
    
    endopar00 = [0.0222456558,0.0050675022,0.1610820404,0.6541901192,0.0555556106 ];
    lb = [   0 ,    0 ,   0 , 0 , 0];
    ub = [0.03 , 0.03 , 1.5 , 1 , 1];
    [endopar,fval] = fminsearchbnd(obj, endopar00, lb, ub,  comp.options);
    fileID = fopen('param_baseline.txt','a');
    fprintf(fileID,'%6.10f %6.10f %6.10f %6.10f %6.10f %6.10f\n',[endopar fval]);
    fclose(fileID);
    
    fprintf('*** Calibrated ***\n')
    
    params = dlmread('param_baseline.txt');
    calibParam = params(end,1:5);
    calibParam(6) = 0;
    calibParam(7) = 0;
end

%% Calibrate extended model with heterogeneity and endogenous search
if calibrate_extended == 1
    targetMom2 = [0.057;0.56;0.9;0.45;0.025;0.23;1];
    
    par.b = 0.7;
    par.xiBad = 0.1;
    func.c_s = @(s,A,psi)  A./(1+psi).*s.^(1+psi);
    func.c_sprime_inv = @(c,A,psi) (c./A).^(1./psi);
    obj2 = @(endopar) calibrationObjective2(endopar, targetMom2, par,func , glob, comp);
    
    fprintf('*** Calibrating extended model ***\n')
        
    endopar00 = [0.0221050241,0.0046065872,0.8000312811,0.8187946759, 1.1395668022, 0.1248277941, 0.3159237974];
    lb = [   0 ,    0 ,   0 , 0 ,   0  , 0, 0];
    ub = [0.03 , 0.03 , 1.5 , 1 , 1.5 , 1, 1];
    [endopar,fval] = fminsearchbnd(obj2, endopar00, lb, ub,  comp.options);
    fileID = fopen('param_extended.txt','a');
    fprintf(fileID,'%6.10f %6.10f %6.10f %6.10f %6.10f %6.10f %6.10f %6.10f\n',[endopar fval]);
    fclose(fileID);
        
    fprintf('*** Calibrated ***\n')
    params = dlmread('param_extended.txt');
    calibParam = params(end,1:7);
end

%% Steady State Solution
par.delta  = calibParam(1);
par.s      = calibParam(2);
par.kappa  = calibParam(3);
par.mu     = calibParam(4);
par.lambda = calibParam(5);
par.pBad   = calibParam(6);
par.pGood  = calibParam(7);

fprintf('*** Solving Steady State ***\n')

ss = solveSS(par, func, glob, comp);

% Vacancy creation cost
par2 = par;
par2.k2 = 1; % baseline model
%par2.k2 = 3.1;  % halving crowdout under baseline model
%par2.k2 = 2.85; % extended model with endogenous search and match het.

fprintf('*** Solved Steady State ***\n')

%% Transiton Paths
if transition == 1
    
    cost = {'c(v)','c(v)','c(n)','c(n)','c(vInflow)','c(vInflow)'};
    repost = [0 , 1 , 0 , 1 , 0 , 1];
    nCase = length(cost);
    
    %% One time, transitory shock to aggregate productivity z.
    
    fprintf('*** Solving Transition: z shock ***\n')
    Tz = 250; % transition duration
    nSigma = 0.015/par.sigmaZ; % size of shock in terms of standard deviations: 1.5% shock
    
    tr = cell(nCase,1);
    for iTr = 1:nCase
        [tr{iTr}] = solveTransition(ss, Tz, nSigma,  par2, func, glob, comp,repost(iTr),cost{iTr}); % reposting
    end
    
    
    fprintf('*** Solved Transition: z shock ***\n')
    plotIRFFigures(100, ss, tr, nSigma,ls)

    [min_u_rep, index_min_u_repost] = min(tr{3}.u);
    min_u_norep = tr{4}.u(index_min_u_repost);
    absDiff = 100* ((min_u_rep - ss.u) - (min_u_norep - ss.u));
    relAmp = 100 * abs( ((min_u_rep - ss.u)./ss.u - (min_u_norep - ss.u)./ss.u) ./ ((min_u_norep - ss.u)./ss.u) );


    %% One time, vacancy drop. Add extra vacancy stock
    fprintf('*** Solving Transition: v shock ***\n')
    Tvac = 250; % transition duration
    tr_vac = cell(nCase,1);
    
    for iTr = 1:nCase
        [tr_vac{iTr}] = solveTransition(ss, Tvac, -2,  par2, func, glob, comp, repost(iTr), cost{iTr});
    end
    
    fprintf('*** Solved Transition: v shock ***\n')
    
    plotIRFFigures(100, ss, tr_vac, -2, ls)
    
    %% One time, lambda shock.
    fprintf('*** Solving Transition: lambda shock ***\n')
    Tlambda = 250; % transition duration
    tr_lambda = cell(nCase,1);
    
    for iTr = 1:nCase
        [tr_lambda{iTr}] = solveTransition(ss, Tvac, -3,  par2, func, glob, comp, repost(iTr), cost{iTr});
    end
    
    fprintf('*** Solved Transition: lambda shock ***\n')
    plotIRFFigures(100, ss, tr_lambda, -3, ls)
    
    %% One time, mu shock.
    fprintf('*** Solving Transition: mu shock ***\n')
    Tmu = 250; % transition duration
    tr_mu = cell(nCase,1);
    
    for iTr = 1:nCase
        [tr_mu{iTr}] = solveTransition(ss, Tvac, -4,  par2, func, glob, comp, repost(iTr), cost{iTr});
    end
    
    fprintf('*** Solved Transition: mu shock ***\n')
    plotIRFFigures(100, ss, tr_mu, -4, ls)
    
    %% One time, beta shock.
    fprintf('*** Solving Transition: beta shock ***\n')
    Tbeta = 250; % transition duration
    tr_beta = cell(nCase,1);
    for iTr = 1:nCase
        [tr_beta{iTr}] = solveTransition(ss, Tvac, -5,  par2, func, glob, comp, repost(iTr), cost{iTr});
    end
    
    fprintf('*** Solved Transition: beta shock ***\n')
    plotIRFFigures(100, ss , tr_beta, -5, ls)
    
    %% Construct comparative statics for k2
    Tvac = 250; % transition duration
    fprintf('*** Computing crowdout ***\n')
    plotCrowdout(ss,Tvac,par2,func,glob,comp,ls)
    fprintf('*** Done ***\n')
end

close all
%% Vacancy multiplier decomposition


delta = par.delta;
gamma = par.gamma;
s = par.s;
lambda = par.lambda;


mSize = 7;
mlt.lineSpec1 = {'lineWidth', 1, 'lineStyle', '-' , 'Color', rgb('DarkGreen'),'Marker','o','MarkerSize',mSize};
mlt.lineSpec2 = {'lineWidth', 1, 'lineStyle', '-', 'Color', rgb('Navy'),'Marker','*','MarkerSize',mSize};
mlt.lineSpec3 = {'lineWidth', 1, 'lineStyle', '-', 'Color', rgb('Maroon'),'Marker','+','MarkerSize',mSize};
mlt.lineSpec4 = {'lineWidth', 1, 'lineStyle', '-' , 'Color', rgb('Black'),'Marker','s','MarkerSize',mSize};

mlt.lineSpec5 = {'lineWidth', 1, 'lineStyle', '-', 'Color', rgb('DarkGreen'),'Marker','d','MarkerSize',mSize};
mlt.lineSpec6 = {'lineWidth', 1, 'lineStyle', '-', 'Color', rgb('Navy'),'Marker','^','MarkerSize',mSize};
mlt.lineSpec7 = {'lineWidth', 1, 'lineStyle', '-',  'Color', rgb('Maroon'),'Marker','.','MarkerSize',mSize};
mlt.lineSpec8 = {'lineWidth', 1, 'lineStyle', '-' , 'Color', rgb('Black'),'Marker','h','MarkerSize',mSize};


TMult = 40;
TMultGrid = 1:TMult;
MarkerIndex1 = 1:7:TMult-6;
MarkerIndex2 = 2:7:TMult-5;
MarkerIndex3 = 3:7:TMult-4;
MarkerIndex4 = 4:7:TMult-3;
MarkerIndex5 = 5:7:TMult-2;
MarkerIndex6 = 6:7:TMult-1;
MarkerIndex7 = 6:7:TMult;


vareps = (0.01*ss.v);

dn_dv = (tr_vac{3}.n-ss.n)/(0.01*ss.v);
dEU_dv = ((1-delta)*gamma*s*([ss.e(1),tr_vac{3}.e(1,1:end-1)]-ss.e(1)))/(0.01*ss.v);
dEE_dv_eFix = ((1-delta)*gamma*(1-s)*lambda*([ss.f,tr_vac{3}.f(1:end-1)]-ss.f)*ss.e(1))/(0.01*ss.v);
dEE_dv_fFix = ((1-delta)*gamma*(1-s)*lambda*ss.f*([ss.e(1),tr_vac{3}.e(1,1:end-1)]-ss.e(1)))/(0.01*ss.v);
dEE_dv_fInter = ((1-delta)*gamma*(1-s)*lambda*([ss.f,tr_vac{3}.f(1:end-1)]-ss.f).*([ss.e(1),tr_vac{3}.e(1,1:end-1)]-ss.e(1)))/(0.01*ss.v);
shock = [1,nan(1,Tvac-1-1)];


dn_dv(abs(dn_dv)<1e-3)=0;
dEU_dv(abs(dEU_dv)<1e-3)=0;
dEE_dv_eFix(abs(dEE_dv_eFix)<1e-3)=0;
dEE_dv_fFix(abs(dEE_dv_fFix)<1e-3)=0;
dEE_dv_fInter(abs(dEE_dv_fInter)<1e-3)=0;


figNum = 1;

figure(figNum)
hold on
plot(TMultGrid,dn_dv(TMultGrid),mlt.lineSpec1{:},'MarkerIndices',MarkerIndex1);
plot(TMultGrid,dEE_dv_eFix(TMultGrid),mlt.lineSpec2{:},'MarkerIndices',MarkerIndex3);
plot(TMultGrid,dEU_dv(TMultGrid)+dEE_dv_fFix(TMultGrid)+dEE_dv_fInter(TMultGrid),mlt.lineSpec3{:},'MarkerIndices',MarkerIndex5);
plot(TMultGrid,dn_dv(TMultGrid)+dEU_dv(TMultGrid)+dEE_dv_eFix(TMultGrid)+dEE_dv_fFix(TMultGrid)+dEE_dv_fInter(TMultGrid),mlt.lineSpec6{:},'MarkerIndices',MarkerIndex6);
hold off 
axis([1 TMult -inf inf])

legend('Only job creation $n$','Only contact rate $f$','Only employment $e$','Total effect'...
    ,'Location','northeast','Orientation','horizontal','NumColumns',1,'FontSize',20)

grid on
set(gca,ls.gridLineSpec{:})
set(gca,ls.fontSpec{:})

title('(a)','FontSize',20)

export_fig(figure(figNum),[ls.graphPath 'vacancy_multipliers'],'-pdf','-transparent');
figNum = figNum + 1;


figure(figNum)
plot(1:(Tvac-1),cumsum(shock,'omitnan'),mlt.lineSpec7{:},'MarkerIndices',MarkerIndex1,'Color', rgb('Gray'),'LineStyle','--','HandleVisibility','off')
hold on
plot(TMultGrid,cumsum(dn_dv(TMultGrid)),mlt.lineSpec1{:},'MarkerIndices',MarkerIndex2);
plot(TMultGrid,cumsum(dEE_dv_eFix(TMultGrid)),mlt.lineSpec2{:},'MarkerIndices',MarkerIndex5);
plot(TMultGrid,cumsum(dEU_dv(TMultGrid)+dEE_dv_fFix(TMultGrid)+dEE_dv_fInter(TMultGrid)),mlt.lineSpec3{:},'MarkerIndices',MarkerIndex5);
plot(TMultGrid,1+cumsum(dn_dv(TMultGrid)+dEU_dv(TMultGrid)+dEE_dv_eFix(TMultGrid)+dEE_dv_fFix(TMultGrid)+dEE_dv_fInter(TMultGrid)),mlt.lineSpec6{:},'MarkerIndices',MarkerIndex7);
hold off


grid on
axis([1 TMult -1 3.5])

set(gca,ls.gridLineSpec{:})
set(gca,ls.fontSpec{:})

title('(c)','FontSize',20)

export_fig(figure(figNum),[ls.graphPath 'vacancy_multipliers_cumul'],'-pdf','-transparent');
figNum = figNum + 1;

%%
ss0 = ss.n + (1-delta)*gamma*(s+(1-s)*lambda*ss.f)*ss.e(1);

irf0 = (tr_vac{3}.n + (1-delta)*gamma*(s+(1-s)*lambda*[ss.f,tr_vac{3}.f(1:end-1)]).*([ss.e(1),tr_vac{3}.e(1,1:end-1)])-ss0)/vareps;
irf1 = (ss.n + (1-delta)*gamma*(s+(1-s)*lambda*[ss.f,tr_vac{3}.f(1:end-1)]).*([ss.e(1),tr_vac{3}.e(1,1:end-1)])-ss0)/vareps; % n at ss
irf2 = (tr_vac{3}.n + (1-delta)*gamma*(s+(1-s)*lambda*ss.f).*([ss.e(1),tr_vac{3}.e(1,1:end-1)])-ss0)/vareps; % f at ss
irf3 = (tr_vac{3}.n + (1-delta)*gamma*(s+(1-s)*lambda*[ss.f,tr_vac{3}.f(1:end-1)]).*ss.e(1)-ss0)/vareps; % e at ss

irf0(abs(irf0)<1e-3)=0;
irf1(abs(irf1)<1e-3)=0;
irf2(abs(irf2)<1e-3)=0;
irf3(abs(irf3)<1e-3)=0;


figure(figNum)
hold on
plot(TMultGrid,irf1(TMultGrid),mlt.lineSpec1{:},'MarkerIndices',MarkerIndex1);
plot(TMultGrid,irf2(TMultGrid),mlt.lineSpec2{:},'MarkerIndices',MarkerIndex2);
plot(TMultGrid,irf3(TMultGrid),mlt.lineSpec3{:},'MarkerIndices',MarkerIndex3);
plot(TMultGrid,irf0(TMultGrid),mlt.lineSpec6{:},'MarkerIndices',MarkerIndex4);
hold off

legend('Constant job creation $n$','Constant contact rate $f$','Constant employment $e$','Total effect',...
    'Location','northeast','NumColumns',1,'FontSize',20);

grid on
set(gca,ls.gridLineSpec{:})
set(gca,ls.fontSpec{:})
axis([1 TMult -inf inf])

title('(b)','FontSize',20)

export_fig(figure(figNum),[ls.graphPath 'vacancy_multipliers_dec'],'-pdf','-transparent');
figNum = figNum + 1;


figure(figNum)
hold on
plot(TMultGrid,1+cumsum(irf1(TMultGrid)),mlt.lineSpec1{:},'MarkerIndices',MarkerIndex1);
plot(TMultGrid,1+cumsum(irf2(TMultGrid)),mlt.lineSpec2{:},'MarkerIndices',MarkerIndex2);
plot(TMultGrid,1+cumsum(irf3(TMultGrid)),mlt.lineSpec3{:},'MarkerIndices',MarkerIndex3);
plot(TMultGrid,1+cumsum(irf0(TMultGrid)),mlt.lineSpec6{:},'MarkerIndices',MarkerIndex4);
hold off
axis([1 TMult -1 3.5])


grid on
set(gca,ls.gridLineSpec{:})
set(gca,ls.fontSpec{:})

title('(d)','FontSize',20)

export_fig(figure(figNum),[ls.graphPath 'vacancy_multipliers_dec_cumul'],'-pdf','-transparent');
figNum = figNum + 1;

