// panelAnalysis
/*
	Exploit the panel structure of the dataset to provide prima facie evidence of
	adjustment costs
	
	Analyses:
	
	1. Do people with earnings near kink at 69 persist there at 70 and 71?
	2. Do people with earnings near kink at 71 have earnings near there at 69?
	3. Persistence of bunching at age 64 kink, look at it in various ways
	
*/

/******************************************************************************/
// Where do people near the kink come from, age 69-73?
/******************************************************************************/

/******************************************************************************/
// Persistence of earnings, bunchers and non-bunchers, 69-73 
/******************************************************************************/
	use "$datadir/ssa_s1Both_micro" if ///
		year>=1990 & inrange(age, 69, 73), clear	
	
	sum realPost if year == 1999
	replace realPost = `r(mean)' if year > 1999 // fix at last value
	
	sort id age
	by id: egen y69 = max( year * (age==69) )
	keep if y69<=1999
	by id: egen t69 = max( real_threshold * (age==69) )
	
	local e = 2.8
	local bsSmall = .8
	local min -10
	local max 20
	local shift = 5.6
	gen ak = abs(real_distance) <= `e'*1e3
	gen ap = abs(real_distance + `shift'*1e3) <= `e'*1e3
	assert ak + ap <= 1
	
	count if ak
	count if ap
	
	sum real_distance if ap
	bysort id (age): egen akSample = max(ak*(age==69))
	bysort id (age): egen apSample = max(ap*(age==69))	
	
	gen count = 1
	drop if real_earnings == 0 |  missing(real_earnings)
	drop if real_se_earnings ~= 0 & ~missing(real_se_earnings)
	keep if akSample | apSample
	
	replace real_distance = (real_earnings - t69) if akSample
	replace real_distance = (real_earnings - t69 + `shift'*1e3) if apSample
	gen rdRound = round(real_distance/1e3, `bsSmall') 		
	
	collapse (sum) count, by(rdRound akSample age)
	bysort akSample age (rdRound): egen tCount = total(count)
	gen prob = count / tCount
	gen dens = prob * 1e3 / `bsSmall'
	gen apSample = ~akSample
	keep if inrange(rdRound, `min', `max')
	
	qui foreach sample in akSample apSample{
		forvalues age = 70/73{
			
			// set up for calculating bunching
			preserve
			keep if age == `age' & `sample'
			
			// fill in zeros in the distribution
			local b0 = round(`min', `bsSmall')
			local bN = round(`max', `bsSmall')
			
			qui forvalues b = `b0'(`bsSmall')`bN'{
				count if abs(rdRound-`b')<1e-6
				if `r(N)' == 0{
					local newObs = _N+1
					set obs `newObs'
					replace rdRound = `b' in `newObs'
					replace prob = 0 in `newObs'
				}
			}
			replace rdRound = round(rdRound, `bsSmall')
			keep if inrange(rdRound, `min', `max')
			
			// calculate bunching			
			local exclude = int(`e'/`bsSmall')
			capture bunchPoly rdRound prob, kink(0) exclude(`exclude') ///
				bw(`bsSmall') degree(7) keep(1) print(0) nboot(100)
			local rc = _rc
			if `rc' ~= 0 { 
				noisily di "error `rc', `age', `sample'"
			}
			local noy "nolabel"
			if "`sample'" == "akSample" local noy ""
			local nox "nolabel"
			if `age' == 73 local nox ""
			local title ""
			if `age' == 70 & "`sample'" == "akSample" local title "A. Bunchers"
			if `age' == 70 & "`sample'" == "apSample" local title "B. Placebo"

			// output results
			if `rc' == 0 {
				
				// print some information about bunching amount
				local b = string(`r(scaledMass)'*`bsSmall', "%5.2f")
				local bSE = string(`r(scaledMassBS)'*`bsSmall', "%5.2f")
				noisily di "age = `age', sample = `sample' b = `b' (`bSE')  "
				gen density = prob / `bsSmall' * 1e3
				replace smoothDensity = smoothDensity /`bsSmall' * 1e3
				
				// graph
				# delimit ;
				twoway 
					(connected density rdRound if ~missing(smoothDensity), sort)
					(line smoothDensity rdRound if ~missing(smoothDensity), sort),
					xlabel(`min'(10)`max', `nox') ylabel(, `noy')
					xline(0, lcolor(gs8))
					graphregion(color(white))
					legend(off)
					title("`title'")
					subtitle("Age `age'", ring(0))
					name(dist`sample'`age', replace)
					xtitle("") ytitle("")
				;
				# delimit cr
				restore
			}
		}
	}
	
	# delimit ;
	graph combine
		distakSample70 distapSample70 
		distakSample71 distapSample71 
		distakSample72 distapSample72 
		distakSample73 distapSample73,
		cols(2) l1title("Density") 
		b1title("Age a distance exempt amount (thousands of dollars)")
		ycommon imargin(small)
		graphregion(color(white))
		xsize(7) ysize(10)
	; 
	# delimit cr
	graph save "$figdir/dynamics7073.gph", replace
	window manage close graph _all
	
	// look at lfp later
	
	

// bunching by age - replicate what we had

	use "$datadir/ssa_s1Both_micro" if ///
		inrange(year, 1990, 1999) ///
		& inrange(age, 57, 73), clear		
	
	drop if real_se_earnings ~= 0 & ~missing(real_se_earnings)
	gen lfp = real_earnings > 0 & ~missing(real_earnings)
	sort id age
	by id: gen lfpL1 = lfp[_n-1]
	by id: gen lfpL2 = lfp[_n-2]
	gen inRange = inrange(real_earnings, -1e4, 3e4)*lfp
	by id: gen inRangeL2 = inRange[_n-2]
	by id: gen inRangeL1 = inRange[_n-1]
	gen allC =lfp*lfpL2*inRange* inRangeL2
	gen all1 =lfp*lfpL1*inRange* inRangeL1
	table age, c(freq sum lfp sum lfpL2 sum inRange sum allC)
			
	merge m:1 year using "$datadir/cpi", assert(2 3) nogen
	sum price_level if year == 2010
	scalar p2010 = r(mean)
	gen threshNom = real_threshold * price_level/ p2010
	gen distanceNom = real_distance * price_level / p2010
	gen earnNom = real_earnings * price_level / p2010
	
	drop if missing(age)
	table year, c(min threshNom max threshNom min real_threshold max real_threshold)
	keep if real_earnings > 0 & ~missing(real_earnings) & earnNom > 25
	gen count = 1
	local bsSmall = 0.56
	local min = -7
	local max = 21
	local e = 2.1
	preserve
	gen rdRound = round(distanceNom / 1e3, `bsSmall')
	
	collapse (sum) count, by(rdRound age)
	bysort age (rdRound): egen tCount = total(count)
	gen prob = count / tCount
	keep if inrange(rdRound, `min', `max')
	
	// continue to exclude 3 bins on either side
	local exclude = int(`e'/`bsSmall')	
	qui forvalues age = 59/73{
	
		local nox "nolabel"
		local noy "nolabel"
		
		if `age' >=69 local nox ""
		if `age' == 59 | `age' == 64 | `age' == 69 local noy ""
	
		bunchPoly rdRound prob if age == `age', kink(0) exclude(`exclude') ///
			bw(`bsSmall') degree(7) keep(1) print(1) nboot(100)
		local b = string(`r(scaledMass)'*`bsSmall', "%5.2f")
		local bSE = string(`r(scaledMassBS)'*`bsSmall', "%5.2f")
		noisily di "age = `age' b = `b' (`bSE')  "

		gen density = prob / `bsSmall' * 1e3
		replace smoothDensity = smoothDensity /`bsSmall' * 1e3
		# delimit ;
		twoway
			(connected density rdRound if !missing(smoothDensity) & age == `age', 
				lcolor(navy) mcolor(navy) msize(tiny))
			(line smoothDensity rdRound if !missing(smoothDensity) & age == `age',  
				lcolor(cranberry)),
			title("`age'", ring(0))
			xlabel(-7(7)21, `nox' labsize(large))
			xtitle("")
			xline(0 `ms', lcolor(gs12))
			ylabel(#4, `noy' labsize(large))
			legend(off)
			graphregion(color(white))
			name(g`age', replace)
		;
		# delimit cr
		capture drop density smoothDensity
	}
	
	# delimit ;
	graph combine 
		g59 g60 g61 g62 g63
		g64 g65 g66 g67 g68
		g69 g70 g71 g72 g73,
		cols(5) imargin(small) ycommon
		b1title("Nominal distance to exempt amount (thousands of dollars)") 
		l1title("Density")
		graphregion(color(white))
	;
	# delimit cr
	graph save "$figdir/bunchingNominalBEPUF.gph", replace
	capture window manage close graph _all
	
	restore
	
	// bunching at age 70, real/nominal and l2 or no
	gen s1 = 1
	gen s2 = allC
	keep if age == 70

	qui foreach s in s1 s2{
		foreach type in real nominal{
			if "`type'" == "nominal" {
				local rd "distanceNom"
				local thresh "threshNom"
				local earn "earnNom"
				local max 30
				local min -10
				local e 2.1
				local bsList .56
			}
			if "`type'" == "real" {
				local rd "real_distance"
				local thresh "real_threshold"
				local earn "real_earnings"
				local max 30
				local min -10
				local e 3
				local bs .8
			}
			preserve
			keep if `s' & inrange(`rd'/1e3, `min', `max')
			gen rdRound = round(`rd'/1e3, `bs')
			collapse (sum) count, by(rdRound)
			
			sum count
			gen prob = count / r(sum)
			
			local exclude = int(`e'/`bs')
			bunchPoly rdRound prob, kink(0) exclude(`exclude') ///
				bw(`bs') degree(7) keep(1) print(0) nboot(100)
			local rc = _rc
			if `rc' ~= 0 { 
				noisily di "error `rc', `s', `type'"
			}
			local noy "nolabel"
			if "`s'" == "s1" local noy ""
			local nox "nolabel"
			if "`type'" == "nominal" local nox ""
		
			// output results
			if `rc' == 0{
			
				// print some information about bunching amount
				local b = string(`r(scaledMass)'*`bsSmall', "%5.2f")
				local bSE = string(`r(scaledMassBS)'*`bsSmall', "%5.2f")
				noisily di "binsize = `binsize', include = `ex' b = `b' (`bSE')  "
				gen density = prob / `bsSmall' * 1e3
				replace smoothDensity = smoothDensity /`bsSmall' * 1e3
				# delimit ;
				twoway
					(connected density rdRound if !missing(smoothDensity), 
						lcolor(navy) mcolor(navy) msize(tiny))
					(line smoothDensity rdRound if !missing(smoothDensity),  
						lcolor(cranberry)),
					title("`s', `type'", ring(0) size(small))
					xlabel(#4, `nox' labsize(large))
					xtitle("")
					xline(0, lcolor(gs12))
					ylabel(#4, `noy' labsize(large))
					legend(off)
					graphregion(color(white))
					name(g`s'`type', replace)
				;
				# delimit cr
			}
			restore
		}
	}
	# delimit ;
	graph combine 
		gs1real gs2real 
		gs1nominal gs2nominal, 
		ycommon
		cols(2) imargin(tiny) l1title("Density") 
		b1title("Age 71 distance to exempt amount")
		graphregion(color(white))
	;
	# delimit cr
	graph save "$figdir/age71.gph", replace
	
/******************************************************************************/
// bunching at the preNRA kink 
/******************************************************************************/

	qui do "$scriptdir\programs\multiBunch.do"
	use "$datadir/ssa_s1Both_micro" if ///
			inrange(year, 1990, 1999) ///
			& inrange(age, 59, 73), clear			
			
	keep if real_earnings > 0 & ~missing(real_earnings) 
	drop if real_se_earnings ~= 0 & ~missing(real_se_earnings)
	sort id age
	local bsSmall = 0.8
	local min = -10
	local max = 30
	local e = 3
	gen dThresh = realPost - realPre
	sum dThresh
	local ms = r(mean) / 1e3
	gen scale = 5600 / dThresh
	sum scale
	
	gen rdRound = round( scale*(real_earnings - realPre)/1e3 , `bsSmall')	
	gen count = 1

	collapse (sum) count, by(rdRound age)
	keep if inrange(rdRound, `min', `max')	
	
	// fill in missing cells
	fillin age rdRound 
	replace count = 0 if _fillin
	bysort age: assert _N == 50
	
	bysort age (rdRound): egen tCount = total(count)
	gen prob = count / tCount
	sum prob 
	gen h = `r(max)'	/`bsSmall' * 1e3	

	//  exclude 2 bins on either side
	local exclude = 2 
	
	gen b0 = .
	gen b0SE = . 
	gen b1 = .
	gen b1SE = .
	forvalues age = 59/73 {
	
		local nox "nolabel"
		local noy "nolabel"
		
		if `age' >=69 local nox ""
		if `age' == 59 | `age' == 64 | `age' == 69 local noy ""
		capture multiBunch rdRound prob ////
			if age == `age' & inrange(rdRound, `min', `max'), ///
			kink(5.6) bw(`bsSmall') ///
			exclude(`exclude') degree(7) keep(1) print(0) nboot(100)
		
		local rc = _rc
		if `rc' ~= 0 {
			noi di "`age' error `rc'"
		}
		if `rc' == 0 {
			replace b0 = `r(scaledMass0)'*`bsSmall' if age==`age'
			replace b0SE = `r(scaledMass0BS)'*`bsSmall'  if age==`age'
			replace b1 = `r(scaledMass1)'*`bsSmall' if age==`age'
			replace b1SE = `r(scaledMass1BS)'*`bsSmall' if age==`age'

			gen density = prob / `bsSmall' * 1e3
			replace smoothDensity = smoothDensity /`bsSmall' * 1e3
			# delimit ;
			twoway
				(area h rdRound if inrange(rdRound, -1.7, 1.7), fcolor(ltblue) lcolor(ltblue))
				(area h rdRound if inrange(rdRound, 3.9, 7.3), fcolor(ltblue) lcolor(ltblue))
				(area h rdRound if inrange(rdRound, -0.1, 0.1), fcolor(gs12) lcolor(gs12))
				(area h rdRound if inrange(rdRound, 5.5, 5.7), fcolor(gs12) lcolor(gs12))
				(connected density rdRound if !missing(smoothDensity) & age == `age', 
					lcolor(navy) mcolor(navy) msize(tiny))
				(line smoothDensity rdRound if !missing(smoothDensity) & age == `age',  
					lcolor(cranberry)),
				title("`age'", ring(0))
				xlabel(-10(10)30, `nox' labsize(large))
				xtitle("")
				ylabel(#4, `noy' labsize(large))
				legend(off)
				graphregion(color(white))
				name(g`age', replace)
			;
			# delimit cr
			capture drop density smoothDensity
		}
	}
	
	# delimit ;
	graph combine 
		g59 g60 g61 g62 g63
		g64 g65 g66 g67 g68
		g69 g70 g71 g72 g73,
		cols(5) imargin(small) ycommon
		b1title("Shifted and scaled earnings distribution (thousands of dollars)") 
		l1title("Density")
		graphregion(color(white))
	;
	# delimit cr
	graph save "$figdir/bunchingPreNRA.gph", replace
	capture window manage close graph _all	
	
	// picture showing both bunching estimates by age, with and without CIs
	sort age rdRound
	egen tagAge = tag(age)
	list age b0 b0SE b1 b1SE if tagAge
	gen upper0 = b0 + 1.96 * b0SE
	gen lower0 = b0 - 1.96 * b0SE
	gen upper1 = b1 + 1.96 * b1SE
	gen lower1 = b1 - 1.96 * b1SE
	# delimit ;
	twoway
		(line upper0 age if tagAge, lc(navy) lp(dash))
		(line lower0 age if tagAge, lc(navy) lp(dash))
		
		(line upper1 age if tagAge, lc(cranberry) lp(dash))
		(line lower1 age if tagAge, lc(cranberry) lp(dash))
		
		(connected b0 age if tagAge, lc(navy) lp(solid))
		(connected b1 age if tagAge, lc(cranberry) lp(solid) mc(cranberry)),
		xline(62 65 70, lcolor(gs8))
		xtitle("Age") ytitle("Scaled excess mass")
		legend(pos(6) cols(2) order(5 "Pre-NRA Kink" 6 "Post-NRA Kink"))
		graphregion(color(white))
	;
	# delimit cr
	graph save "$figdir/bunchBothKinksTSCI.gph", replace
	
	# delimit ;
	twoway
		(connected b0 age if tagAge, lc(navy) lp(solid))
		(connected b1 age if tagAge, lc(cranberry) lp(solid) mc(cranberry)),
		xline(62 65 70, lcolor(gs8))
		xtitle("Age") ytitle("Scaled excess mass")
		legend(pos(6) cols(2) order(1 "Pre-NRA Kink" 2 "Post-NRA Kink"))
		graphregion(color(white))
	;
	# delimit cr
	graph save "$figdir/bunchBothKinksTS.gph", replace
		
/******************************************************************************/
// 1 Persistence of bunching at age 69 kink (at ages 70 and 71)
/******************************************************************************/
	use "$datadir/ssa_s1Both_micro" if ///
		inrange(year, 1990, 1999) ///
		& inrange(age, 69, 71), clear			
		
	merge m:1 year using "$datadir/cpi", assert(2 3) nogen
	sum price_level if year == 2010
	scalar p2010 = r(mean)
	gen distanceNom = real_distance * price_level / p2010
	gen earnNom = real_earnings * price_level / p2010
	gen threshNom = real_threshold * price_level / p2010
	
	local type "real"
	if "`type'" == "nominal" {
		local rd "distanceNom"
		local thresh "threshNom"
		local earn "earnNom"
		local max 30
		local min -10
		local e 2.1
		local bsList 280 560 1120
		local exList 0.7(0.7)2.8 21
	}
	if "`type'" == "real" {
		local rd "real_distance"
		local thresh "real_threshold"
		local earn "real_earnings"
		local max 30
		local min -10
		local e 3
		local bsList 400 800 1600
		local exList 1/4 30
	}
	
	// missing se income, but ok here		
	sort id age
	by id: gen rdLag1 = `rd'[_n-1]
	by id: gen rdLag2 = `rd'[_n-2]
	by id: gen threshLag1 = `thresh'[_n-1]
	by id: gen threshLag2 = `thresh'[_n-2]
	gen keeper = real_earnings > 0 & ~missing(real_earnings) 
	replace keeper = 0 if real_se_earnings ~= 0 & ~missing(real_se_earnings)
	
	by id: gen keeperL1 = keeper[_n-1]
	by id: gen keeperL2 = keeper[_n-2]
	gen count = 1	
	
	tab age
	tab age if real_earnings > 0 & ~missing(real_earnings)	
	gen keep70 = keeper & keeperL1 & year>=1989
	gen keep71 = keeper & keeperL2
	keep if keep70 | keep71 
		
	qui foreach a of numlist 71 70{
		foreach ex of numlist `exList'{
			local n  = 0 
			foreach binsize of numlist `bsList'{
				local ++n
				local bsSmall = `binsize'/1e3
				preserve
				
				// define bunchers 
				if `a' == 70 local L "1"
				if `a' == 71 local L "2"
				gen ak = abs(rdLag`L')/1e3 <= `ex'			
				*gen rdRound = round( (`earn' - threshLag`L')/1e3, `bsSmall')
				gen rdRound = round( `rd'/1e3, `bsSmall')
				keep if ak & inrange(rdRound, `min', `max') & age == `a'
				collapse (sum) count, by(rdRound)
				sum count
				gen prob = count / r(sum)
				
				// fill in zeros in the distribution
				local b0 = round(`min', `bsSmall')
				local bN = round(`max', `bsSmall')
				
				qui forvalues b = `b0'(`bsSmall')`bN'{
					count if abs(rdRound-`b')<1e-6
					if `r(N)' == 0{
						local newObs = _N+1
						set obs `newObs'
						replace rdRound = `b' in `newObs'
						replace prob = 0 in `newObs'
					}
				}
				replace rdRound = round(rdRound, `bsSmall')
				keep if inrange(rdRound, `min', `max')
				
				// look at bunching and earnings distribution
				local exclude = int(`e'/`bsSmall')
				capture bunchPoly rdRound prob, kink(0) exclude(`exclude') ///
					bw(`bsSmall') degree(7) keep(1) print(0) nboot(100)
				local rc = _rc
				if `rc' ~= 0 { 
					noisily di "error `rc', `ex', `binsize'"
				}
				local noy "nolabel"
				if `ex'==1 local noy ""
				local nox "nolabel"
				if `n' == 3 local nox ""
			
				// output results
				if `rc' == 0{
				
					// print some information about bunching amount
					local b = string(`r(scaledMass)'*`bsSmall', "%5.2f")
					local bSE = string(`r(scaledMassBS)'*`bsSmall', "%5.2f")
					noisily di "binsize = `binsize', include = `ex' b = `b' (`bSE')  "
					gen density = prob / `bsSmall' * 1e3
					replace smoothDensity = smoothDensity /`bsSmall' * 1e3
					# delimit ;
					twoway
						(connected density rdRound if !missing(smoothDensity), 
							lcolor(navy) mcolor(navy) msize(tiny))
						(line smoothDensity rdRound if !missing(smoothDensity),  
							lcolor(cranberry)),
						title("{&delta}=`binsize', K=`ex'", ring(0) size(small))
						xlabel(#4, `nox' labsize(large))
						xtitle("")
						xline(0 `ms', lcolor(gs12))
						ylabel(#4, `noy' labsize(large))
						legend(off)
						graphregion(color(white))
						name(g`ex'`n', replace)
					;
					# delimit cr
				}
						
				// restore
				restore 
				
			}
		}
		
		// combine the graphs
		#  delimit ; 
		graph combine
			g11 g21 g31 g41 g301
			g12 g22 g32 g42 g302
			g13 g23 g33 g43 g303, ycommon
			cols(5) imargin(tiny) l1title("Density") 
			b1title("Age `a' `type' distance to age `a' exempt amount")
			graphregion(color(white))
		;
		# delimit cr
		graph save "$figdir/dist`a'AK69.gph", replace 
		capture window manage close graph _all
	}
	
		
	
/******************************************************************************/
// XX Earnings growth COP and not
/******************************************************************************/
	use "$datadir/ssa_s1Both_micro" if ///
		inrange(year, 1990, 1999) ///
		& inrange(age, 69, 71), clear		
		
	drop if real_se_earnings ~= 0 & ~missing(real_se_earnings)
	replace real_earnings = 0 if missing(real_earnings)
	sort id age
	by id: gen g1 = (real_earnings[_n+1] - real_earnings) / real_earnings
	by id: gen g1COP = g1 if real_earnings[_n+1]>0
  by id: gen g2 = (real_earnings[_n+2] - real_earnings) / real_earnings
  by id: gen g2COP = g2 if real_earnings[_n+2]>0
  
	foreach x in 1 1COP 2 2COP{
		gen sd`x' = g`x'
	}
		
	gen rdRound = round(real_distance, 800)/1e3
	gen count = 1
	keep if real_earnings > 0 
	collapse (sum) count (mean) g1-g2COP (sd) sd1-sd2COP, by(rdRound age)
	
	gen upper = .
	gen lower = .
	
	// age 69-70 growth
	foreach suff in "" "COP"{
		replace upper = g1`suff' + 1.96*sd1`suff'/sqrt(count)
		replace lower= g1`suff' - 1.96*sd1`suff'/sqrt(count)
		local mod ""
		if "`suff'" == "COP" local mod " (COP)"
		# delimit ;
		twoway
			(connected g1`suff' rdRound if age == 69 & abs(rdRound)<=10)
			(line upper rdRound if age == 69 & abs(rdRound)<=10, lcolor(cranberry))
			(line lower rdRound if age == 69 & abs(rdRound)<=10, lcolor(cranberry)),
			graphregion(color(white))
			title("Age 69 to 70`mod'", ring(0))
			xtitle("") ytitle("")
			xlabel(-10(5)10, nolabel) 
			legend(off) xline(0, lcolor(gs12))
			name(g6970`suff', replace)
		;
		# delimit cr		
	}
	
	// age 69-71 growth
	foreach suff in "" "COP"{
		replace upper = g2`suff' + 1.96*sd1`suff'/sqrt(count)
		replace lower = g2`suff' - 1.96*sd1`suff'/sqrt(count)
		local mod ""
		if "`suff'" == "COP" local mod " (COP)"
		# delimit ;
		twoway
			(connected g2`suff' rdRound if age == 69 & abs(rdRound)<=10)
			(line upper rdRound if age == 69 & abs(rdRound)<=10, lcolor(cranberry))
			(line lower rdRound if age == 69 & abs(rdRound)<=10, lcolor(cranberry)),
			graphregion(color(white))
			title("Age 69 to 71`mod'", ring(0))
			xtitle("") ytitle("")
			xlabel(-10(5)10, nolabel) 
			legend(off) xline(0, lcolor(gs12))
			name(g6971`suff', replace)
		;
		# delimit cr		
	}
	
	// age 70-71 growth 
	foreach suff in "" "COP"{
		replace upper = g1`suff' + 1.96*sd1`suff'/sqrt(count)
		replace lower= g1`suff' - 1.96*sd1`suff'/sqrt(count)
		local mod ""
		if "`suff'" == "COP" local mod " (COP)"
		# delimit ;
		twoway
			(connected g1`suff' rdRound if age == 70 & abs(rdRound)<=10)
			(line upper rdRound if age == 70 & abs(rdRound)<=10, lcolor(cranberry))
			(line lower rdRound if age == 70 & abs(rdRound)<=10, lcolor(cranberry)),
			graphregion(color(white))
			title("Age 70 to 71`mod'", ring(0))
			xtitle("") ytitle("")
			xlabel(-10(5)10) 
			legend(off) xline(0, lcolor(gs12))
			name(g7071`suff', replace)
		;
		# delimit cr		
	}
	
	# delimit ;
	graph combine 
		g6970 g6970COP
		g6971 g6971COP
		g7071 g7071COP, cols(2)
		graphregion(color(white))
		l1title("Percent change in earnings")
		b1title("Base age distance to exempt amount")
		imargin(small)
	;
	# delimit cr
	graph save "$figdir/earningsGrowth6971", replace
	capture window manage close graph _all
		
/******************************************************************************/
// 2 Where do people who bunch at age 71 come from? 
/******************************************************************************/

	// not totally clear which age range we should be using, but whatever
	use "$datadir/ssa_s1Both_micro" if ///
		inrange(year, 1990, 1999) ///
		& inrange(age, 69, 71), clear		
	
	xtset id age
	gen rdLag1 = l.real_distance
	gen rdLag2 = ll.real_distance
	
	gen keeper = real_earnings > 0 & ~missing(real_earnings)
	replace keeper = 0 if real_se_earnings ~= 0 & ~missing(real_se_earnings)
	gen keeperL1 = l.keeper
	gen keeperL2 = ll.keeper
	keep if keeper & (keeperL1 | keeperL2) &  (age == 71 | age == 70)
	gen count = 1
	
	foreach a of numlist 70 71 {
		if `a' == 70 local L "1"
		if `a' == 71 local L "2"
		noi di "`L'"
		qui foreach ex of numlist  1/4 30 {
			foreach binsize of numlist 400 800 1600 {
				local bsSmall = `binsize'/1e3
				
				// preserve for looping
				preserve
				// define bunchers
				gen ak = abs(real_distance)/1e3 <= `ex'
				
				keep if ak & age == `a' & keeperL`L'
				gen rdRound = round(rdLag`L'/1e3, `bsSmall')
				keep if inrange(rdRound, -10, 30)
				collapse (sum) count, by(rdRound)
				sum count
				gen prob = count / r(sum)
				
				// fill in zeros in the distribution
				local b0 = round(-10, `bsSmall')
				local bN = round(30, `bsSmall')
				
				qui forvalues b = `b0'(`bsSmall')`bN'{
					count if abs(rdRound-`b')<1e-6
					if `r(N)' == 0{
						local newObs = _N+1
						set obs `newObs'
						replace rdRound = `b' in `newObs'
						replace prob = 0 in `newObs'
					}
				}
				replace rdRound = round(rdRound, `bsSmall')
				keep if inrange(rdRound, -10, 30)
				
				// cfop bunching amount and graph		
				local exclude = int(3/`bsSmall')
				capture bunchPoly rdRound prob, kink(0) exclude(`exclude') ///
					bw(`bsSmall') degree(7) keep(1) print(0) nboot(100)
				local rc = _rc
				if `rc' ~= 0 { 
					noisily di "`rc', `ex', `binsize'"
				}
				local noy "nolabel"
				if `ex'==1 local noy ""
				local nox "nolabel"
				if `binsize' == 1600 local nox ""
			
				// output results
				if `rc' == 0{
				
					// print some information about bunching amount
					local b = string(`r(scaledMass)'*`bsSmall', "%5.2f")
					local bSE = string(`r(scaledMassBS)'*`bsSmall', "%5.2f")
					noisily di "age `a', binsize = `binsize', include = `ex' b = `b' (`bSE')  "
					gen density = prob / `bsSmall' * 1e3
					replace smoothDensity = smoothDensity /`bsSmall' * 1e3
					# delimit ;
					twoway
						(connected density rdRound if !missing(smoothDensity), 
							lcolor(navy) mcolor(navy) msize(tiny))
						(line smoothDensity rdRound if !missing(smoothDensity),  
							lcolor(cranberry)),
						title("{&delta}=`binsize', K=`ex'", ring(0) size(small))
						xlabel(#4, `nox' labsize(large))
						xtitle("")
						xline(0 `ms', lcolor(gs12))
						ylabel(#4, `noy' labsize(large))
						legend(off)
						graphregion(color(white))
						name(g`ex'`binsize', replace)
					;
					# delimit cr
				}
						
				
				
				// restore
				restore 
			}
		}

		// combine the graphs
		#  delimit ; 
		graph combine
			g1400 g2400 g3400 g4400 g30400
			g1800 g2800 g3800 g4800 g30800
			g11600 g21600 g31600 g41600 g301600, ycommon
			cols(5) imargin(tiny) l1title("Density") 
			b1title("Scaled age 69 distance to exempt amount")
			graphregion(color(white))
		;
		# delimit cr
		graph save "$figdir/dist69AK`a'.gph", replace 
		capture window manage close graph _all
	}
		
/******************************************************************************/
// 3 Persistence of bunching at age 64 kink
/******************************************************************************/

	use "$datadir/ssa_s1Both_micro" if ///
		inrange(year, 1989, 1999) ///
		& inrange(age, 64, 66), clear		
			
	sort id age
	by id: gen rdLag1 = real_distance[_n-1]
	by id: gen rdLag2 = real_distance[_n-2]
	by id: gen threshLag1 = real_threshold[_n-1]
	gen keeper = real_earnings > 0 & ~missing(real_earnings)
	replace keeper = 0 if real_se_earnings ~= 0 & ~missing(real_se_earnings)
	by id: gen keeperL1 = keeper[_n-1]
	by id: gen keeperL2 = keeper[_n-2]
	keep if keeper 
	gen count = 1
	gen tt = real_threshold - threshLag1
	sum tt if age == 65
	local ms = r(mean) / 1e3
	gen scale = `ms' / (real_threshold - threshLag1) 
	gen rdShift = (real_earnings - threshLag1)*scale

	sum rdShift if abs(real_distance)<=1e3 & age == 65
	sum real_distance if abs(rdShift - `ms') <= 1 & age == 65
	
	qui foreach ex of numlist  1/4 30 {
		foreach binsize of numlist 1600 400 800 {
			local bsSmall = `binsize'/1e3
			
			// preserve for looping
			preserve
			// define bunchers
			gen ak = abs(rdLag1)/1e3 <= `ex'
			
			keep if ak	& age == 65	& year > = 1990 & keeperL1
			gen rdRound = round(rdShift, `bsSmall')
			keep if inrange(rdRound, -10, 30)
			collapse (sum) count, by(rdRound)
			sum count
			gen prob = count / r(sum)
			
			// fill in zeros in the distribution
			local b0 = round(-10, `bsSmall')
			local bN = round(30, `bsSmall')
			
			qui forvalues b = `b0'(`bsSmall')`bN'{
				count if abs(rdRound-`b')<1e-6
				if `r(N)' == 0{
					local newObs = _N+1
					set obs `newObs'
					replace rdRound = `b' in `newObs'
					replace prob = 0 in `newObs'
				}
			}
			replace rdRound = round(rdRound, `bsSmall')
			keep if inrange(rdRound, -10, 30)
			
			// cfop bunching amount and graph		
			local exclude = int(3/`bsSmall')
			capture bunchPoly rdRound prob, kink(0) exclude(`exclude') ///
				bw(`bsSmall') degree(7) keep(1) print(0) nboot(100)
			local rc = _rc
			if `rc' ~= 0 { 
				noisily di "`rc', `ex', `binsize'"
			}
			local noy "nolabel"
			if `ex'==1 local noy ""
			local nox "nolabel"
			if `binsize' == 1600 local nox ""
		
			// output results
			if `rc' == 0{
			
				// print some information about bunching amount
				local b = string(`r(scaledMass)'*`bsSmall', "%5.2f")
				local bSE = string(`r(scaledMassBS)'*`bsSmall', "%5.2f")
				noisily di "binsize = `binsize', include = `ex' b = `b' (`bSE')  "
				gen density = prob / `bsSmall' * 1e3
				replace smoothDensity = smoothDensity /`bsSmall' * 1e3
				# delimit ;
				twoway
					(connected density rdRound if !missing(smoothDensity), 
						lcolor(navy) mcolor(navy) msize(tiny))
					(line smoothDensity rdRound if !missing(smoothDensity),  
						lcolor(cranberry)),
					title("{&delta}=`binsize', K=`ex'", ring(0) size(small))
					xlabel(#4, `nox' labsize(large))
					xtitle("")
					xline(0 `ms', lcolor(gs12))
					ylabel(#4, `noy' labsize(large))
					legend(off)
					graphregion(color(white))
					name(g`ex'`binsize', replace)
				;
				# delimit cr
			}
					
			
			
			// restore
			restore 
		}
	}

	// combine the graphs
	#  delimit ; 
	graph combine
		g1400 g2400 g3400 g4400 g30400
		g1800 g2800 g3800 g4800 g30800
		g11600 g21600 g31600 g41600 g301600, ycommon
		cols(5) imargin(tiny) l1title("Density") 
		b1title("Scaled age 65 distance to age 64 exempt amount")
		graphregion(color(white))
	;
	# delimit cr
	graph save "$figdir/dist65AK64.gph", replace 
	capture window manage close graph _all
	
/******************************************************************************/
// 3.5 Age 66 earnings distribution of age 64 bunchers
/******************************************************************************/

	qui foreach ex of numlist  1/4 30 {
		foreach binsize of numlist 1600 400 800 {
			local bsSmall = `binsize'/1e3
			
			// preserve for looping
			preserve
			// define bunchers
			gen ak = abs(rdLag2)/1e3 <= `ex'
			
			keep if ak	& age == 66	& year > = 1990 & keeperL2
			gen rdRound = round(real_distance/1e3, `bsSmall')
			keep if inrange(rdRound, -10, 30)
			collapse (sum) count, by(rdRound)
			sum count
			gen prob = count / r(sum)
			
			// fill in zeros in the distribution
			local b0 = round(-10, `bsSmall')
			local bN = round(30, `bsSmall')
			
			qui forvalues b = `b0'(`bsSmall')`bN'{
				count if abs(rdRound-`b')<1e-6
				if `r(N)' == 0{
					local newObs = _N+1
					set obs `newObs'
					replace rdRound = `b' in `newObs'
					replace prob = 0 in `newObs'
				}
			}
			replace rdRound = round(rdRound, `bsSmall')
			keep if inrange(rdRound, -10, 30)
			
			// cfop bunching amount and graph		
			local exclude = int(3/`bsSmall')
			capture bunchPoly rdRound prob, kink(0) exclude(`exclude') ///
				bw(`bsSmall') degree(7) keep(1) print(0) nboot(100)
			local rc = _rc
			if `rc' ~= 0 { 
				noisily di "`rc', `ex', `binsize'"
			}
			local noy "nolabel"
			if `ex'==1 local noy ""
			local nox "nolabel"
			if `binsize' == 1600 local nox ""
		
			// output results
			if `rc' == 0{
			
				// print some information about bunching amount
				local b = string(`r(scaledMass)'*`bsSmall', "%5.2f")
				local bSE = string(`r(scaledMassBS)'*`bsSmall', "%5.2f")
				noisily di "binsize = `binsize', include = `ex' b = `b' (`bSE')  "
				gen density = prob / `bsSmall' * 1e3
				replace smoothDensity = smoothDensity /`bsSmall' * 1e3
				# delimit ;
				twoway
					(connected density rdRound if !missing(smoothDensity), 
						lcolor(navy) mcolor(navy) msize(tiny))
					(line smoothDensity rdRound if !missing(smoothDensity),  
						lcolor(cranberry)),
					title("{&delta}=`binsize', K=`ex'", ring(0) size(small))
					xlabel(#4, `nox' labsize(large))
					xtitle("")
					xline(0 `ms', lcolor(gs12))
					ylabel(#4, `noy' labsize(large))
					legend(off)
					graphregion(color(white))
					name(g`ex'`binsize', replace)
				;
				# delimit cr
			}
					
			
			
			// restore
			restore 
		}
	}

	// combine the graphs
	#  delimit ; 
	graph combine
		g1400 g2400 g3400 g4400 g30400
		g1800 g2800 g3800 g4800 g30800
		g11600 g21600 g31600 g41600 g301600, ycommon
		cols(5) imargin(tiny) l1title("Density") 
		b1title("Age 66 distance to exempt amount")
		graphregion(color(white))
	;
	# delimit cr
	graph save "$figdir/dist66AK64.gph", replace 
	capture window manage close graph _all
	
/******************************************************************************/
// 4 Where do age 65 bunchers come from?
/******************************************************************************/

	qui foreach ex of numlist  1/4 30 {
		foreach binsize of numlist 400 800 1600{
			local bsSmall = `binsize'/1e3
			
			// preserve for looping
			preserve
			// define bunchers
			gen ak = abs(real_distance)/1e3 <= `ex'
			
			keep if ak	& age == 65 	& year > = 1990 & keeperL1
			gen rdRound = round(rdLag1, `binsize')/1e3
			keep if inrange(rdRound, -10, 30)
			
			collapse (sum) count, by(rdRound)
			sum count
			gen prob = count / r(sum)
			
			// fill in zeros in the distribution
			local b0 = round(-10, `bsSmall')
			local bN = round(30, `bsSmall')
			qui forvalues b = `b0'(`bsSmall')`bN'{
				count if abs(rdRound-`b')<1e-6
				if `r(N)' == 0{
					local newObs = _N+1
					set obs `newObs'
					replace rdRound = `b' in `newObs'
					replace prob = 0 in `newObs'
				}
			}
			replace rdRound = round(rdRound, `bsSmall')
			keep if inrange(rdRound, -10, 30)
			// cfop bunching amount and graph		
			local exclude = int(3/`bsSmall')
			capture bunchPoly rdRound prob, kink(0) exclude(`exclude') ///
				bw(`bsSmall') degree(7) keep(1) print(0) nboot(100)
			local rc = _rc
		
			local noy "nolabel"
			if `ex' == 1 local noy ""
			local nox "nolabel"
			if `binsize' == 1600 local nox ""
		
			// output results
			if `rc' == 0{
			
				// print some information about bunching amount
				local b = string(`r(scaledMass)'*`bsSmall', "%5.2f")
				local bSE = string(`r(scaledMassBS)'*`bsSmall', "%5.2f")
				noisily di "binsize = `binsize', include = `ex' b = `b' (`bSE')  "
				gen density = prob / `bsSmall'
				replace smoothDensity = smoothDensity /`bsSmall'
				# delimit ;
				twoway
					(connected density rdRound if !missing(smoothDensity), 
						lcolor(navy) mcolor(navy) msize(tiny))
					(line smoothDensity rdRound if !missing(smoothDensity),  
						lcolor(cranberry)),
					title("{&delta}=`binsize', K=`ex'", ring(0) size(small))
					xlabel(#4, `nox' labsize(large))
					xtitle("")
					xline(0, lcolor(gs12))
					ylabel(#4, `noy' labsize(large))
					legend(off)
					graphregion(color(white))
					name(g`ex'`binsize', replace)
				;
				# delimit cr
			}
					
			
			
			// restore
			restore 
		}
	}

	// combine the graphs
	#  delimit ; 
	graph combine
		g1400 g2400 g3400 g4400 g30400
		g1800 g2800 g3800 g4800 g30800
		g11600 g21600 g31600 g41600 g301600, ycommon
		cols(5) imargin(tiny) l1title("Density") 
		b1title("Age 64 distance to exempt amount")
		graphregion(color(white))
	;
	# delimit cr
	graph save "$figdir/dist64AK65.gph", replace 
	capture window manage close graph _all

	
/******************************************************************************/
// 5 Where do age 66 bunchers come from (age 64)?
/******************************************************************************/

	qui foreach ex of numlist  1/4 30 {
		foreach binsize of numlist 400 800 1600{
			local bsSmall = `binsize'/1e3
			
			// preserve for looping
			preserve
			// define bunchers
			gen ak = abs(real_distance)/1e3 <= `ex'
			
			keep if ak	& age == 66	& year > = 1990 & keeperL2
			gen rdRound = round(rdLag2, `binsize')/1e3
			keep if inrange(rdRound, -10, 30)
			
			collapse (sum) count, by(rdRound)
			sum count
			gen prob = count / r(sum)
			
			// fill in zeros in the distribution
			local b0 = round(-10, `bsSmall')
			local bN = round(30, `bsSmall')
			qui forvalues b = `b0'(`bsSmall')`bN'{
				count if abs(rdRound-`b')<1e-6
				if `r(N)' == 0{
					local newObs = _N+1
					set obs `newObs'
					replace rdRound = `b' in `newObs'
					replace prob = 0 in `newObs'
				}
			}
			replace rdRound = round(rdRound, `bsSmall')
			keep if inrange(rdRound, -10, 30)
			// cfop bunching amount and graph		
			local exclude = int(3/`bsSmall')
			noi di "`exclude'"
			capture bunchPoly rdRound prob, kink(0) exclude(`exclude') ///
				bw(`bsSmall') degree(7) keep(1) print(0) nboot(100)
			local rc = _rc
			noi di "`ex', `binsize', `rc'"
			local noy "nolabel"
			if `ex' == 1 local noy ""
			local nox "nolabel"
			if `binsize' == 1600 local nox ""
		
			// output results
			if `rc' == 0{
			
				// print some information about bunching amount
				local b = string(`r(scaledMass)'*`bsSmall', "%5.2f")
				local bSE = string(`r(scaledMassBS)'*`bsSmall', "%5.2f")
				noisily di "binsize = `binsize', include = `ex' b = `b' (`bSE')  "
				gen density = prob / `bsSmall'
				replace smoothDensity = smoothDensity /`bsSmall'
				# delimit ;
				twoway
					(connected density rdRound if !missing(smoothDensity), 
						lcolor(navy) mcolor(navy) msize(tiny))
					(line smoothDensity rdRound if !missing(smoothDensity),  
						lcolor(cranberry)),
					title("{&delta}=`binsize', K=`ex'", ring(0) size(small))
					xlabel(#4, `nox' labsize(large))
					xtitle("")
					xline(0, lcolor(gs12))
					ylabel(#4, `noy' labsize(large))
					legend(off)
					graphregion(color(white))
					name(g`ex'`binsize', replace)
				;
				# delimit cr
				
			}
			
			// restore
			restore 
		}
	}

	// combine the graphs
	#  delimit ; 
	graph combine
		g1400 g2400 g3400 g4400 g30400
		g1800 g2800 g3800 g4800 g30800
		g11600 g21600 g31600 g41600 g301600, ycommon
		cols(5) imargin(tiny) l1title("Density") 
		b1title("Age 64 distance to exempt amount")
		graphregion(color(white))
	;
	# delimit cr
	graph save "$figdir/dist64AK66.gph", replace 
	capture window manage close graph _all

	
/******************************************************************************/
// 6 Simpler tabular depiction
/******************************************************************************/
	
	use "$datadir/ssa_s1Both_micro" if ///
		inrange(year, 1990, 1999) ///
		& inrange(age, 64, 66), clear		
	
	gen noLFP = real_earnings == 0 | missing(real_earnings) ///
		| (real_se_earnings ~= 0 & ~missing(real_se_earnings))
	gen ak = abs(real_distance)<=2e3 & ~noLFP
	gen above = real_distance > 2e3 & ~noLFP 
	gen below = real_distance < -2e3  & ~noLFP
	
	gen cat = 1 if noLFP
	replace cat = 2 if below
	replace cat = 3 if ak
	replace cat = 4 if above
	assert ~missing(cat)
	
	xtset id age
	gen catL2 = ll.cat
	gen catL1 = l.cat
	keep if inrange(year, 1992, 1997)
	tab cat catL2 if age == 66
	
	qui forvalues age = 65/66 {
		count if age == `age'
		gen tXc0 = "Category &" in 1
		local N = `r(N)'
		local l = `age'-64
		forvalues catCol = 1/5{
			if `catCol' == 1 local cName "Exit"
			if `catCol' == 2 local cName "Below"
			if `catCol' == 3 local cName "At kink"
			if `catCol' == 4 local cName "Above kink"
			if `catCol' == 5 local cName "Total"
			
			gen tXc`catCol' = "`cName' &" in 1
			local row = 1
			forvalues catRow = 1/5{
				if `catRow' == 1 local rName "Exit"
				if `catRow' == 2 local rName "Below"
				if `catRow' == 3 local rName "At kink"
				if `catRow' == 4 local rName "Above kink"
				if `catRow' == 5 local rName "Total"
				
				if `catCol' <5 {
					if `catRow' < 5 {
						count if catL`l' == `catRow' & cat == `catCol' & age == `age'
						local ++row
						replace tXc0 = "`rName' &" in `row'
						replace tXc`catCol'= "`r(N)' &" in `row'
						local ++row
						replace tXc`catCol' = "["+string(100*`r(N)'/`N', "%4.2f") + "] &" in `row'
						
					}
					if `catRow' == 5 {
						count if cat == `catCol' & age == `age'
						local ++row
						replace tXc0 = "`rName' &" in `row'
						replace tXc`catCol' = "`r(N)' &" in `row'
						local ++row
						replace tXc`catCol'= "["+string(100*`r(N)'/`N', "%4.2f") + "] &" in `row'
					
					}
				}
				if `catCol' == 5 {
					if `catRow' < 5 {
						count if catL`l' == `catRow' & age == `age'
						local ++row
						replace tXc0 = "`rName' &" in `row'
						replace tXc`catCol' = "`r(N)' &" in `row'
						local ++row
						replace tXc`catCol' = "["+string(100*`r(N)'/`N', "%4.2f") + "] &" in `row'
						
					}
					if `catRow' == 5 {
						count if age == `age'
						local ++row
						replace tXc0 = "`rName' &" in `row'
						replace tXc`catCol' = "`r(N)' &" in `row'
						local ++row
						replace tXc`catCol' = "["+string(100*`r(N)'/`N', "%4.2f") + "] &" in `row'
					
					}
				}
			
			}
		}
		
		noi list tXc* in 1/`row', noobs clean 
		noi di "" ""
		replace tXc5 = subinstr(tXc5, "&", "\\", .)
		replace tXc5 = tXc5 + "\midrule" in 1
		replace tXc5 = tXc5 + "\midrule" in 9
		outsheet tXc* in 1/`row' using "$tabdir/crossTab64`age'.tex", ///
			replace noquote nonames delimit(" ")
		drop tXc*
	}
	
/******************************************************************************/
// X placebo test at Xk below age 64 kink 
/******************************************************************************/

	use "$datadir/ssa_s1Both_micro" if ///
		inrange(year, 1989, 1999) ///
		& inrange(age, 64, 66), clear
	
	local dp = -4e3
	gen placeboThresh = real_threshold + `dp'
	gen placeboDist = real_earnings - placeboThresh
	sort id age
	by id: gen rdLag1 = placeboDist[_n-1]
	by id: gen threshLag1 = placeboThresh[_n-1]
	by id: gen realThreshLag1 = real_threshold[_n-1]
	by id: gen earnLag1 = real_earnings[_n-1]
	gen keeper = real_earnings > 0 & ~missing(real_earnings)
	by id: gen keeperL1 = keeper[_n-1]
	keep if keeperL1 & keeper 
	gen count = 1
	gen tt = real_threshold - realThreshLag1
	sum tt if age == 65
	local ms = r(mean) /1e3
	gen scale = `ms' / (real_threshold - realThreshLag1)  
	
	gen rdShift = (real_earnings - threshLag1 )*scale

	sum rdShift if abs(real_distance)<=1e3 & age == 65
	sum real_distance if abs(rdShift - `ms') <= 1 & age == 65
			
	qui foreach ex of numlist  1/4 30 {
		foreach binsize of numlist 400 800 1600{
			local bsSmall = `binsize'/1e3
			
			// preserve for looping
			preserve
			// define bunchers
			gen ak = abs(rdLag1 )/1e3 <= `ex'
			
			keep if ak	& age == 65	& year > 1989
			gen rdRound = round(rdShift, `bsSmall')
			keep if inrange(rdRound, -5, 25)
			collapse (sum) count, by(rdRound)
			sum count
			gen prob = count / r(sum)
			
			// fill in zeros in the distribution
			local b0 = round(-5, `bsSmall')
			local bN = round(15, `bsSmall')
			
			qui forvalues b = `b0'(`bsSmall')`bN'{
				count if abs(rdRound-`b')<1e-6
				if `r(N)' == 0{
					local newObs = _N+1
					set obs `newObs'
					replace rdRound = `b' in `newObs'
					replace prob = 0 in `newObs'
				}
			}
			replace rdRound = round(rdRound, `bsSmall')
			keep if inrange(rdRound, -5, 15)
			// cfop bunching amount and graph		
			local exclude = int(3/`bsSmall')
			capture bunchPoly rdRound prob, kink(0) exclude(`exclude') ///
				bw(`bsSmall') degree(7) keep(1) print(0) nboot(100)
			local rc = _rc
			
			if `rc' ~= 0 {
				noi di "`binsize' `ex' `rc'"
			}
		
			local noy "nolabel"
			if `ex'==1 local noy ""
			local nox "nolabel"
			if `binsize' == 1600 local nox ""
		
			// output results
			if `rc' == 0{
			
				// print some information about bunching amount
				local b = string(`r(scaledMass)'*`bsSmall', "%5.2f")
				local bSE = string(`r(scaledMassBS)'*`bsSmall', "%5.2f")
				noisily di "binsize = `binsize', include = `ex' b = `b' (`bSE')  "
				gen density = prob / `bsSmall'
				replace smoothDensity = smoothDensity /`bsSmall'
				# delimit ;
				twoway
					(connected density rdRound if !missing(smoothDensity), 
						lcolor(navy) mcolor(navy) msize(tiny))
					(line smoothDensity rdRound if !missing(smoothDensity),  
						lcolor(cranberry)),
					title("{&delta}=`binsize', K=`ex'", ring(0) size(small))
					xlabel(#4, `nox' labsize(large))
					xtitle("")
					xline(0 `ms', lcolor(gs12))
					ylabel(#4, `noy' labsize(large))
					legend(off)
					graphregion(color(white))
					name(g`ex'`binsize', replace)
				;
				# delimit cr
			}
					
			
			
			// restore
			restore 
		}
	}

	// combine the graphs
	#  delimit ; 
	graph combine
		g1400 g2400 g3400 g4400 g30400
		g1800 g2800 g3800 g4800 g30800
		g11600 g21600 g31600 g41600 g301600, ycommon
		cols(5) imargin(tiny) l1title("Density") 
		b1title("Scaled age 65 distance to age 64 exempt amount")
		graphregion(color(white))
	;
	# delimit cr
	graph save "$figdir/placeboDist65AK64.gph", replace 
	capture window manage close graph _all

	
/******************************************************************************/
// X2 Age time series of bunching at pre-NRA kink and at post-NRA kink 
/******************************************************************************/
	local bs = 800
	local bsSmall = `bs'/1e3
	local e = 3
	qui foreach thresh in Pre Post {
		use "$datadir/ssa_s1Both_micro" if ///
			inrange(age, 59, 73) & inrange(year, 1990, 1999) & ///
			real_earnings > 0 & real_earnings < 6e4 & ///
			(real_se_earnings == 0 | missing(real_se_earnings) ), clear
		
		// distance to relevant thrsehold
		gen rdRound= round( real_earnings - real`thresh' , `bs')/1e3
		keep if inrange(rdRound, -10, 30)
		
		// collapse to histograms 
		gen count = 1
		collapse (sum) count, by(rdRound age)
		bysort age (rdRound): egen tCount = total(count)
		gen prob = count / tCount
		gen dens = prob * 1e3 / `bsSmall'
		
		// calculate bunching at each age
		noisily di ""
		noisily di "Threshold = `thresh'"
		forvalues a = 59/73 {
			local exclude = int(`e'/`bsSmall')
			capture bunchPoly rdRound prob if age == `a' , ////
				kink(0) exclude(`exclude') ///
				bw(`bsSmall') degree(7) keep(1) print(0) nboot(100)
				
			// if error, say so
			local rc = _rc
			if `rc' ~= 0 { 
				noisily di "  `age' error `rc'"
			}		
			if `rc' == 0 {
				local b = string(`r(scaledMass)'*`bsSmall', "%5.2f")
				local bSE = string(`r(scaledMassBS)'*`bsSmall', "%5.2f")
				noisily di "  `a' `b' (`bSE')"
			}
			
			// plot for 65 year-olds, pre
			if `rc' == 0 & `a' == 65 & "`thresh'" == "Pre" {
				replace smoothDensity = smoothDensity /`bsSmall' * 1e3
				gen density = prob / `bsSmall' * 1e3
				# delimit ;
				twoway
					(connected density rdRound if !missing(smoothDensity) & age==`a', 
						lcolor(navy) mcolor(navy) msize(tiny))
					(line smoothDensity rdRound if !missing(smoothDensity) & age==`a',  
						lcolor(cranberry)),
					xtitle("Age 65 distance to pre-NRA exempt amount")
					ytitle("Density")
					xline(0 5.6, lcolor(gs8))
					legend(off)
					graphregion(color(white))
				;
				# delimit cr
				graph save "$figdir/age65DistPre.gph", replace

			}
		}
	}
		
	
