function[gamma, IES, flag] = labor_est (h0, hf, wr, fix_h, fix_c)

% This function solves for the value of gamma that satisfies the
% indifference condition (Equation (9)), given the number of hours under the fixed
% schedule (h0) and the flexible schedule (hf), the ratio between the
% reservation wages for the fixed and the flexible schedule (wr), and the
% calibrated fixed costs of work (fix_h, fix_c). 

% Auxiliary terms used to calculate Equation (9). 
l0 = 1-fix_h-h0;
lf = 1-fix_h-hf*h0;
RHS = (h0*(1-fix_c))/(h0 - h0*hf*wr);


options = optimoptions('fsolve','Display','off','TolX',1.0e-6);

 % 0: Estimate obtained
 % 1: No interior solution. Demand a high wage for working less hours, which cannot be
 % justified by the calibrated fixed cost of work.  
 % Imply a very high IES under a larger fixed cost of work 
 % 2: No interior solution. Wanting to work very little, which cannot be
 % justified by the calibrated fixed cost of work.  
 % Imply a very small IES under a smaller fixed cost of work
 % 3: Irrational response.  Demand more salary when working less. 
 % 4: Identical responses to fixed and flexible schedule scenarios 

if (hf<1 && h0 < hf*h0*wr) || (hf>1 && h0 > hf*h0*wr)
    x = 0; flag = 3; % Irrational response.  Demand more salary when working less. 
elseif (hf==1 && h0 ~= hf*h0*wr) || (hf~=1 && h0 == hf*h0*wr)
    x = 0; flag = 3; % Irrational response.  Demand more salary when working less. 
elseif (hf==1 && h0 == hf*h0*wr)
    x = 0; flag = 4; % Identical responses to fixed and flexible schedule scenarios 
elseif RHS > ((fix_h + h0)/(h0 - h0*hf))
    x = 0; flag = 1; %No interior solution. Demand a high wage for working less hours, which cannot be
                     % justified by the calibrated fixed cost of work.  
                     % Imply a very high IES under a larger fixed cost of work 
elseif RHS<1
    x=0; flag = 2;   % 2: No interior solution. Wanting to work very little, which cannot be
                     % justified by the calibrated fixed cost of work.  
                     % Imply a very small IES under a smaller fixed cost of work
else
    x = fsolve(@(x) RHS - (1-l0^(1-1/x))/(lf^(1-1/x)-l0^(1-1/x)), 0.5, options);
    flag = 0;  % Solve for the value of gamma that satisfies equation (9)
end

gamma = x;
% Transform gamma to IES
IES = gamma * (1-h0)/h0;



