function spcsim = simSpecCI(R,N,TT,TT2,kr)
%
% This function obtains N bootstrapped smoothed spectra using data drawn
% from a DGP characterized by the autocovariance function R.
% 
% Inputs:
%   R           autocovariance function; R(j) = Cov(X_t,X_(t-j+1))
%   N           number of simulated spectra to obtain
%   TT          number of frequencies in two-sided spectrum
%   TT2         number of frequencies in one-sided spectrum
%   kr          kernel function for smoothing
%
% Outputs:
%   spcsim      spcsim(:,j) is j-th simulated one-sided spectrum
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%% Set-up

T = numel(R);                   % length of a simulated sample
nkr = numel(kr);                % size of kernel

spcsim = zeros(TT2,N);          % final array to hold all simulated spectra

Sig = toeplitz(R);              % covariance matrix associated with autocovariance function R
cSig = chol(Sig,'lower');       % Cholesky decomposition of Sig

% we do the simulation in blocks of 10,000 to avoid memory issues
bsz = 10000;
k = floor(N/bsz);               % number of whole 10,000-simulation blocks
spcsimj = zeros(TT2,bsz,k);     % array to hold only the full 10,000-simulation blocks

%% Do the full bsz-simulation blocks

for j = 1:k
    simdat = cSig*randn(T,bsz);         % bsz random draws from series with autocovariance function R
    
    pfflp = abs(fft(simdat,TT)).^2;     % un-scaled two-sided spectrum for each draw
    pfflp = pfflp(1:TT2,:)/(2*pi*T);    % scaled one-sided spectrum for each draw
    
    conspec = [pfflp(nkr:-1:2,:);pfflp;pfflp(end-1:-1:end-nkr+1,:)]; % extend for wrap-around kernel-smoothing
    conspec = conv2(conspec,kr,'same');             % kernel-smooth
    spcsimj(:,:,j) = conspec(nkr:TT2+nkr-1,:);      % drop extra frequencies added for wrap-around
end

% reshape array containing the k bsz-simulation blocks
if k > 0    % if at least one full 10,000-simulation block was done
    spcsim(:,1:bsz*k) = reshape(spcsimj,TT2,bsz*k);
end

%% Do extra simulations needed to get to N total

if N > k*bsz
    simdat = cSig*randn(T,N-k*bsz);  % remaining random draws from series with autocovariance function R
    
    pfflp = abs(fft(simdat,TT)).^2;     % un-scaled two-sided spectrum
    pfflp = pfflp(1:TT2,:)/(2*pi*T);    % scaled one-sided spectrum
    
    conspec = [pfflp(nkr:-1:2,:);pfflp;pfflp(end-1:-1:end-nkr+1,:)]; % extend for wrap-around kernel-smoothing
    conspec = conv2(conspec,kr,'same');                 % kernel-smooth
    spcsim(:,bsz*k+1:end) = conspec(nkr:TT2+nkr-1,:);    % drop extra frequencies added for wrap-around
end 
