function [rpr,Sig] = fnRegrChisq(rcind,kvc,x)
%
% This function computes point estimates and Newey-West covariance matrix
% estimate for recession probabilities at horizons in kvc.
% 
% Inputs:
%   rcind       vector of recession indicators
%   kvc         vector of horizons k to test
%   x           window around t+k (must be scalar)
%
% Outputs:
%   rpr         point estimates for recession probabilities at each horizon
%   Sig         covariance matrix of point estimates across horizons
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%% Set-up

kmx = max(kvc);         % max horizon length
T = numel(rcind);       % number of periods in sample

nk = numel(kvc);        % number of horizons

wn = ones(2*x+1,1);     % vector of ones equal to total window size (2x+1)

% t-th element of rcwn is 1 iff date t has a recession at some point in
% x-quarter window around it
rcwn = min(conv(rcind,wn,'same'),1);

Tk = T-kmx-x;           % last date t of sample for which entire t+kmx+/-x 
                            % window is in the data
rcindk = rcind(1:Tk);   % consider only these elements of rcind
X = [ones(Tk,1),rcindk];    % matrix of regressors

% This block constructs the matrix y of regressands
y = zeros(Tk,nk);   % q-th column of y corresponds to horizon kvc(q)
for q = 1:nk        % for each value of k
    
    k = kvc(q);         
    
    % take corresponding elements of rcwn, i.e, t-th element of y(:,q)
        % corresponds to (t+kvc(q))-th element of rcwn
    y(:,q) = rcwn(k+1:k+Tk);
        
end


%% compute recession probabilities and covariance matrix estimate

e = ones(2,1);

bet = (X'*X)\X'*y;      % bet(:,q) gives regression point estimates for
                            % horizon kvc(q)
rpr = bet'*e;           % rpr(q) gives recession probability point estimate
                            % for horizon kvc(q)

uall = y - X*bet;       % matrix of residuals

% Letting p denote vector of true cond'l recession probabilities (of which
% rpr is an estimate), one can show that
% 	T*Cov(rpr-p) -> sum_(j=-infty:infty) E[Z_t' Z_(t-j)] as T -> infy,
% where
%   Z_t =  e' (E[X(t,:) X(t,:)'])^(-1) X(t,:) uall(t,:)
% The following obtains the Newey-West estimate of E[Z_t' Z_(t-j)].
Z = zeros(Tk,nk);       % allocate matrix, where Z(t,:) = Z_t
Phi = e'/(X'*X/Tk);        % quantity to be used repeatedly
for t = 1:Tk
    Z(t,:) = Phi*X(t,:)'*uall(t,:); % Z_t
end

Qall = NewWest(Z,0,1);  % get Newey-West estimate of E[Z_t' Z_(t-j)]
Sig = Qall/Tk;          % divide by Tk to get estimate of Cov(rpr-p)

