% This file is for conducting hypothesis tests for the presence of a peak
% in the spectrum against the null of either a locally flat or AR(1)
% spectrum. It requires the file Data.mat (see ReadMe.pdf for description).
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%% Initialize

clearvars

%% Options and parameters

% These parameters control lower and upper bounds for peak and trough
% ranges. Only Fourier frequencies (FFs) contained in these ranges will be
% considered as part of a peak/trough.
pkrg = [32,40];         % peak range bounds (incl. both bounds)
trrg1 = [16,32];        % lower trough range bounds (incl. only lower bound)
trrg2 = [pkrg(2),60];   % upper trough range bounds (incl. only upper bound)

Nsim = 1000000;         % number of Monte Carlo draws
sd = 1546034615;        % seed for random number generator (for consistency)

rng(sd);

%% Data options

flnm = 'Data';      % file name with data in it

% The file Data.mat contains a single struct variable called Data. Each
% field of this struct variable is another struct variable corresponding to
% a different data series (see ReadMe.pdf for list of data series). The
% fields for each variable are as follows:
%   data:       vector of actual data
%   LongName:   longer description of data series
%   fdat:       first quarter of data available for that series
%   ldat:       last quarter of data available for that series
%   m100:       flag = 1 if the data should be multiplied by 100 to convert
%                   from fractions/logs to percentages

xname = 'lhnfbBLS';     % name of data series to use; see ReadMe.pdf

% Date ranges to use; if range exceeds available range for data, code
% throws a warning and uses the maximum range available.
estfst = 0;           % first observation to use
estlst = 2015.25;     % last observation to use


%% Read in data and de-mean

load(flnm)      % load file containing data (also contains dates of
                % observations in the variable ddate)

x = Data.(xname).data;      % extract vector of data
m100 = Data.(xname).m100;   % extract flag = 1 if need to multiply by 100
fdat = Data.(xname).fdat;   % extract first available quarter of data
ldat = Data.(xname).ldat;   % extract last available quarter of data

dat = (fdat:.25:ldat)';     % vector of available quarters

fdat1 = estfst;             % variable to hold first quarter to use
ldat1 = estlst;             % variable to hold last quarter to use

% This block checks whether desired first quarter precedes available first
% quarter. If so, fdat1 is changed to first available quarter and a warning
% is reported.
if fdat > estfst
    yr = floor(fdat);
    qr = 4*(fdat-yr)+1;
    qrt = [num2str(yr) 'Q' num2str(qr)];
    warning(['First date precedes earliest in data. Starting at ' qrt '.'])
    fdat1 = fdat;
end
% This block checks whether desired last quarter is after available last
% quarter. If so, ldat1 is changed to last available quarter and  a warning
% is reported.
if ldat < estlst
    yr = floor(ldat);
    qr = 4*(ldat-yr)+1;
    qrt = [num2str(yr) 'Q' num2str(qr)];
    warning(['Last date after latest in data. Ending at ' qrt '.'])
    ldat1 = ldat;
end

kpind = (dat>=fdat1) & (dat<=ldat1);    % indices of quarters to keep

xc = x(kpind);          % extract subsample

% if desired, multiply by 100
if m100 == 1
    xc = 100*xc;
end

xc = xc-mean(xc);       % de-mean sample
T = numel(xc);          % length of sample


%% Periodogram of data

% This block determines the number of points in one sided of the spectrum
Todd = logical(mod(T,2));       % flag = true if TT is odd
if Todd                         % if TT is odd
    TT2 = (T+1)/2;                        	% number of points in one side of fft
    freqs = linspace(0,.5*(T-1)/T,TT2)';    % vector of ordinary frequencies, evenly spaced
else                            % otherwise if TT is even
    TT2 = T/2 + 1;                          % number of points in one side of fft
    freqs = linspace(0,.5,TT2)';            % vector of ordinary frequencies, evenly spaced
end

I = abs(fft(xc,T)).^2/(2*pi*T);     % scaled two-sided spectrum
I = I(1:TT2);                       % one-sided spectrum

%% Get information about peak/trough 

% indices of peak and trough ranges
pkind = (1./freqs>=pkrg(1))&(1./freqs<=pkrg(2));        
trind1 = ((1./freqs>=trrg1(1))&(1./freqs<trrg1(2)));
trind2 = ((1./freqs>trrg2(1))&(1./freqs<=trrg2(2)));

% numbers of Fourier frequencies in each range
mP = nnz(pkind);
mT1 = nnz(trind1);
mT2 = nnz(trind2);

%% Set up null hypotheses

% In what follows, null j=1 corresponds to the flat null and null j=2 to
% the AR(1) null. Both nulls are modeled as an AR(1) process, where the
% flat null has an autocorrelation of 0.

sig2 = zeros(1,2);      % sig2(j) is innovation variance associated with null j
rho = zeros(1,2);       % rho(j) is autocorrelation associated with null j

% flat null
rho(1) = 0;
sig2(1) = var(xc);

% AR(1) null
rho(2) = corr(xc(1:end-1),xc(2:end));
sig2(2) = var(xc(2:end)-rho(2)*xc(1:end-1));


g = 1+rho.^2-2*rho.*cos(2*pi*freqs);    % function in denominator of null spectrum
f = sig2./(2*pi*g);                     % null spectrum


%% Compute Monte Carlo p-values under nulls

pvls = zeros(1,2);      % vector to hold output

dfpk = 2*mP;            % d.o.f. for peak range
dftr1 = 2*mT1;          % d.o.f. for lower trough range
dftr2 = 2*mT2;          % d.o.f. for upper trough range

% draw values from null distribution for D_P, D_T1, D_T2 (see text for
% details)
D_P_sim = chi2rnd(dfpk,Nsim,1)/mP;
D_T1_sim = chi2rnd(dftr1,Nsim,1)/mT1;
D_T2_sim = chi2rnd(dftr2,Nsim,1)/mT2;

% Monte Carlo draw of test statistics under null
F1_sim = D_P_sim./D_T1_sim;
F2_sim = D_P_sim./D_T2_sim;

% values of test statistics in data
R = 2*I./f;
D_P = mean(R(pkind,:),1);
D_T1 = mean(R(trind1,:),1);
D_T2 = mean(R(trind2,:),1);
F1 = D_P./D_T1;
F2 = D_P./D_T2;

for k = 1:2     % for each null, compute fraction of Monte Carlo draws
                    % where both test statitistics are at least as great as
                    % those observed in the data
    pvls(1,k) = nnz((F1_sim > F1(k))&(F2_sim > F2(k)))/Nsim;
end


%% Create Output

res = cell(1,3);                % cell array for display output
datnm = Data.(xname).LongName;  % long name of data series

% create output table for display, as well as Latex code
res{1,2} = 'Flat';
res{1,3} = 'AR(1)';
res{2,1} = datnm;
ltx = datnm;
for k = 1:2         % for each null
    
    pki = 100*pvls(1,k);            % convert p-values to percentages
    res{2,1+k} = num2str(pki,2);    % load p-value into output cell array
    
    % set display precision for Latex (1 decimal place for p-values < 10%,
    % no decimal places for p-values > 10%)
    if pki > 10
        frmt = '%.0f';
    else
        frmt = '%.1f';
    end
    ltx = [ltx ' & ' num2str(pki,frmt) '\%'];   % latex code
end
ltx = [ltx ' \\'];

% print output
disp(datnm)
disp(res)
disp(ltx)










