function [expns,cmbs] = TaylExp(N,ord)
% This function returns a matrix 'expns' representing the monomials in a
% 'ord'-th-order multivariate polynomial (MP) in 'N' variables. Each row of
% 'expns' is of the form [n1,...,nN], and represents the monomial
%
%       x1^n1 * x2^n2 * ... * xN^nN
%
% The number of times each monomial appears in a Taylor expansion is given
% in the vector 'cmbs' (e.g., in a second-order Taylor expansion, x1*x2
% appears as d(df/dx1)/dx2 and d(df/dx2)/dx1, while x1^2 only appears once
% as d(df/dx1)/dx1).
%
% Written by: Dana Galizia, Carleton University
% Last edit: 2016-05-30

% Total number of terms in the N-th-degree MP in the
% state before eliminating duplicate terms
ntrmall = sum(N.^(1:ord));

% Dummy vector to hold terms of an N-th-degree MP in
% the state before eliminating duplicate terms
dumtrm = zeros(ntrmall,1);

% Cell matrix, j-th element of which to hold indices of j-th-degree terms
% of MP in state before eliminating duplicate terms
incll = cell(ord,1);

% Here, one prime number is associated with each state variable. For any two numbers
% x1 and x2 obtained as products of k1 and k2 of these primes (using the same prime more
% than once is allowed), respectively, x1 = x2 if and only if they have the same set of
% prime factors with the same multiplicities, i.e., if and only if they are associated
% with the same MP term.
tmp = (1+N)^2;
suffprm = false;
while ~suffprm
    prms = primes(tmp)';          % generate a vector of at least nst prime #s
    if numel(prms) >= N
        suffprm = true;
    else
        tmp = 2*tmp;
    end
end
prms = prms(1:N);                 % take the first nst primes, each associated with a state var

% first-degree terms 
incll{1} = (1:N)';         % indices of first-degree terms
dumtrm(incll{1}) = prms;   % first section of dummy vector contains first-degree terms

fjind = N;    % variable to track last-used index of dummy vector
nnew = N;     % variable to track number of new terms at current polynomial order
% loop over degrees from 2 to N
for j = 2:ord
    nnew = N*nnew;                            % number of j-th degree terms
    incll{j} = (fjind+1:fjind+nnew)';    % indices of j-th degree terms
    tmp = dumtrm(incll{j-1})*prms';    % to get j-th degree terms, multiply (j-1)-th
                                                % degree terms by prms
    dumtrm(incll{j}) = tmp(:);         % assign to dummy vector
    fjind = fjind + nnew;                       % update last-used index
end

% Get indices for first appearance of a given polynomial term in dummy vector,
% then sort these indices in ascending order. Given any vector computed as
% dumYNall but with the actual state in place of prms, selpol gives indices of
% unique elements.
[~,seltrm] = unique(dumtrm,'first');
seltrm = sort(seltrm);
ntrm = numel(seltrm);         % number of unique elements

dumkp = dumtrm(seltrm);
expns = zeros(ntrm,N);
cmbs = zeros(ntrm,1);

for j = 1:ntrm
    prmfct = factor(dumkp(j));
    for k = 1:N
        expns(j,k) = sum(prmfct==prms(k));
    end
    cmbs(j) = sum(dumtrm==dumkp(j));
end




