function [s,F] = DatSpec(Y,varargin)

% [s,F] = DatSpec(Y,[kr],[dm],[TT])
%
% This function non-parametrically estimates the spectrum of the autocovariance
% function for the data, then optionally smoothes across frequencies using a
% provided kernel.
%
% This function is mostly just a wrapper for MATLAB's built-in fft function, with
% the following added benefits:
%
%   (1) Spectra and cross-spectra for an arbitrary number of data series can be computed
%       with one function call.
%   (2) The output is appropriately scaled to get one side of a two-sided spectrum,
%       so that integrating the output across angular frequencies between 0 and pi
%       then adding the result to its conjugate tranpose will yield the unconditional
%       covariance matrix for the data (.
%   (3) Frequency-averaged smoothing is possible (built-in MATLAB functions appear only to
%       do ensemble averages, such as Welch's method).
% 
%
% Inputs:
%
%       Y:      T-by-n          ,   matrix of data series; rows are observations, columns
%                                   are different series
%       [kr]:   nkr-by-1        ,   optional kernel function for smoothing
%       [dm]:   scalar          ,   1 = first de-mean Y (default)
%       [TT]:   scalar          ,   number of points at which to get fft;
%                                       default is T
%
% Outputs:
%
%       s:      n-by-n-by-TT2   ,   s(i,m,j) is cross-spectrum between i-th and m-th
%                                   variables at j-th frequency
%       F:      TT2-by-1        ,   list of frequencies
%
% Note: If a kernel is provided, it should have an odd number of entries. The
%       function assumes that the middle entry is the center of the kernel.

% Written by: Dana Galizia, Carleton University
% Last edit: 2016-05-28


%% initial set-up

if nargin < 2
    kr = [];
else
    kr = varargin{1};
end
nkr = numel(kr);
smth = (nkr>1);       % logical flag indicating whether smoothing is necessary

if nargin < 3
    dm = true;
else
    dm = varargin{2};
    if numel(dm) == 0
        dm = true;
    else
        dm = logical(dm(1));
    end
end

[T,n] = size(Y);

if nargin < 4
    TT = T;
else
    TT = varargin{3};
    if numel(TT) == 0
        TT = T;
    else
        TT = TT(1);
    end
end

Todd = logical(mod(TT,2));

if Todd
    TT2 = (TT+1)/2;                         % number of points in one side of fft
    F = linspace(0,.5*(TT-1)/TT,TT2);      % vector of ordinary frequencies, evenly spaced
else
    TT2 = TT/2 + 1;                 % number of points in one side of fft
    F = linspace(0,.5,TT2)';        % vector of ordinary frequencies, evenly spaced
end

if smth
    if size(kr,2) ~= 1
        kr = kr';
    end
    if size(kr,2) ~=1
        error('Kernel should be a vector.')
    end
    if nkr/2 == round(nkr/2)
        error('Kernel should contain odd # of entries.')
    end
    if nkr > TT2
        error('Kernel is too long.')
    end
    kr = kr/sum(kr);
end



s = zeros(n,n,TT2);

%% De-mean if necessary

if dm
    Y = Y-repmat(mean(Y),T,1);
end

%% Compute spectrum

% Fourier transforms
fftmt = zeros(TT,n);
for j = 1:n
    fftmt(:,j) = fft(Y(:,j),TT);
end

% spectra/cross-spectra
for j = 1:n
    for k = j:n
        tmp = (fftmt(1:TT2,j).*conj(fftmt(1:TT2,k)))/(2*pi*T);
%         tmp = (fftmt(1:TT2,j).*conj(fftmt(1:TT2,k)))/(2*pi*TT);
        if smth
            conspec = [tmp(nkr:-1:2);tmp;tmp(end-1:-1:end-nkr+1)];
            conspec = conv(conspec,kr,'same');
            tmp = conspec(nkr:TT2+nkr-1);
        end
        s(j,k,:) = reshape(tmp,1,1,TT2);
        if k > j
            s(k,j,:) = conj(s(j,k,:));
        end
    end
end


