* compug.dta is a Stata file downloaded via WRDS from the Compustat Global Security Daily dataset
* variables (in addition to those that come automatically): ISIN SEDOL EXCHG TPCI GSUBIND AJEXDI CSHOC PRCCD TRFD CURCDD LOC 
* Dates: 1/1/2016 to 11/1/2017

* euexchg.dta is a Stata file that identifies Compustat exchange codes that correspond to stock exchanges located in Europe

* isoexrates.dta is a Stata file with daily currency exchange rates.  Ultimate source is https://www.federalreserve.gov/releases/h10/Hist/


set more off
clear

local dataprep = 1
local indices = 1
local table_a3 = 1
local graphs = 1

if `dataprep' == 1 {

	use compug, clear
	keep tpci isin gvkey iid datadate prccd curcdd ajexdi trfd loc gsubind exchg sedol cshoc

	/* Keep only common stocks with ISIN codes (latter helps eliminate duplicate issues) */
	keep if tpci=="0"
	drop if isin==""

	rename datadate d

	/* Keep only the primary issue */
	sort gvkey iid d
	qby gvkey: keep if iid==iid[1]

	/* Generate codes identifying European stocks using ISIN, HQ LOC, Exchange, and Currency */
	gen isin2=substr(isin,1,2)

	gen byte euloc=0 if loc~=""
	replace euloc=1 if loc=="DEU"|loc=="FRA"|loc=="NLD"|loc=="ITA"|loc=="ESP"|loc=="FIN"|loc=="PRT"|loc=="GRC"|loc=="IRL"|loc=="BEL"|loc=="LUX"|loc=="AUT"|loc=="SVN"|loc=="SVK"|loc=="LVA"|loc=="LTU"|loc=="EST"|loc=="CYP"|loc=="MLT"
	replace euloc=2 if loc=="GBR"|loc=="DNK"|loc=="SWE"|loc=="POL"|loc=="HUN"|loc=="CZE"|loc=="ROU"|loc=="BGR"|loc=="HRV"
	replace euloc=3 if loc=="ISL"|loc=="CHE"|loc=="LIE"|loc=="NOR"

	gen byte euisin=0 if isin2~=""
	replace euisin=1 if isin2=="DE"|isin2=="FR"|isin2=="NL"|isin2=="IT"|isin2=="ES"|isin2=="FI"|isin2=="PT"|isin2=="GR"|isin2=="IE"|isin2=="BE"|isin2=="LU"|isin2=="AT"|isin2=="SK"|isin2=="SI"|isin2=="LT"|isin2=="LV"|isin2=="EE"|isin2=="CY"|isin2=="MT"
	replace euisin=2 if isin2=="GB"|isin2=="DK"|isin2=="SE"|isin2=="PL"|isin2=="HU"|isin2=="CZ"|isin2=="RO"|isin2=="BG"|isin2=="HR"
	replace euisin=3 if isin2=="IS"|isin2=="CH"|isin2=="LI"|isin2=="NO"

	label var euisin "=1 if Eurozone, =2 if EU, = 3 if EEA"

	sort exchg
	merge exchg using euexchg, nokeep
	drop _merge

	label var euexchg "=1 if Eurozone, =2 if EU, = 3 if EEA"

	gen byte eucur = 0 if curcdd~=""
	replace eucur = 1 if curcdd=="EUR"
	replace eucur = 2 if curcdd=="GBP"|curcdd=="SEK"|curcdd=="DKK"|curcdd=="PLN"|curcdd=="HUF"|curcdd=="CZK"|curcdd=="RON"|curcdd=="BGN"|curcdd=="HRK"
	replace eucur = 3 if curcdd=="ISK"|curcdd=="CHF"|curcdd=="NOK"

	/* Keep only European stocks */
	keep if eucur>0&eucur~=.
	keep if euisin>0&euisin~=.
	keep if euloc>0&euloc~=.
	keep if euexchg>0&euexchg~=.

	/* Merge with exchange rate data*/
	sort curcdd d
	merge curcdd d using isoexrates, nokeep
	drop _merge

	save compug_europe, replace
	
	}

if `indices' == 1 {
	
	use compug_europe, clear
	local zeroday = mdy(6,23,2016)
	keep if d>=`zeroday'

	/* If dividend gross-up factor missing assume no dividends */
	egen mtrfd=mean(trfd), by(gvkey iid)
	replace trfd=1 if mtrfd==.
	drop mtrfd

	/* Create return and prior-day market-cap */
	sort gvkey iid d
	qby gvkey iid: replace rate=rate[_n-1] if rate==.&(rate[_n+1]~=.|rate[_n+2]~=.|rate[_n+3]~=.)
	qby gvkey iid: gen mcap=cshoc[_n-1]*prccd[_n-1]/rate[_n-1]
	qby gvkey iid: gen ret=(trfd*prccd/ajexdi/rate)/(trfd[_n-1]*prccd[_n-1]/ajexdi[_n-1]/rate[_n-1])-1
		
	gen temp=ret if d==mdy(6,24,2016)
	su temp [aw=mcap]
	replace temp=temp-r(mean)

	egen ret624=mean(temp), by(gvkey)
	drop temp

	/* Generate L-S index based on relative performance on Brexit day */
	gen w=ret624*mcap

	gen xy=ret*w if w>0
	gen x=w if ret~=.&w>0
	egen sumxy=sum(xy), by(d)
	egen sumx=sum(x), by(d)
	gen retl=sumxy/sumx
	drop xy x sumxy sumx
	
	gen xy=ret*w if w<0
	gen x=w if ret~=.&w<0
	egen sumxy=sum(xy), by(d)
	egen sumx=sum(x), by(d)
	gen rets=sumxy/sumx
	drop xy x sumxy sumx

	gen retls=retl-rets
	
	if `table_a3' == 1 {
		
		gen gsector=substr(gsubind,1,2)
		log using table_a3, t replace
		tab gsector if d==mdy(6,24,2016)&w>0&w~=. [aw=mcap]
		tab gsector if d==mdy(6,24,2016)&w<0&w~=. [aw=mcap]
		tab loc if d==mdy(6,24,2016)&w>0&w~=. [aw=mcap]
		tab loc if d==mdy(6,24,2016)&w<0&w~=. [aw=mcap]
		su ret if d==mdy(6,24,2016)&w>0&w~=. [aw=mcap]
		su ret if d==mdy(6,24,2016)&w<0&w~=. [aw=mcap]
		log close
		}
		
	gen ggrp=substr(gsubind,1,4)
	gen uk=(curcdd=="GBP")

	/* UK only index */
	gen w_uk=ret624*mcap if uk==1

	gen xy=ret*w_uk if w_uk>0
	gen x=w_uk if ret~=.&w_uk>0
	egen sumxy=sum(xy), by(d)
	egen sumx=sum(x), by(d)
	gen retl_uk=sumxy/sumx
	drop xy x sumxy sumx

	gen xy=ret*w_uk if w_uk<0
	gen x=w_uk if ret~=.&w_uk<0
	egen sumxy=sum(xy), by(d)
	egen sumx=sum(x), by(d)
	gen rets_uk=sumxy/sumx
	drop xy x sumxy sumx

	gen retls_uk=retl_uk-rets_uk


	/* Between index (industry and UK/non) */
	gen xy=ret*mcap if d==mdy(6,24,2016)
	gen x=mcap if ret~=.&d==mdy(6,24,2016)
	egen sumxy=sum(xy), by(uk ggrp)
	egen sumx=sum(x), by(uk ggrp)
	gen ret624_ukggrp=sumxy/sumx
	drop xy x sumxy sumx
	su ret624_ukggrp if d==mdy(6,24,2016) [aw=mcap]
	replace ret624_ukggrp=ret624_ukggrp-r(mean)
	gen w_ukggrp=(ret624_ukggrp)*mcap

	gen xy=ret*w_ukggrp if w_ukggrp>0 
	gen x=w_ukggrp if ret~=.&w_ukggrp>0
	egen sumxy=sum(xy), by(d)
	egen sumx=sum(x), by(d)
	gen retl_ukggrp=sumxy/sumx
	drop xy x sumxy sumx

	gen xy=ret*w_ukggrp if w_ukggrp<0 
	gen x=w_ukggrp if ret~=.&w_ukggrp<0
	egen sumxy=sum(xy), by(d)
	egen sumx=sum(x), by(d)
	gen rets_ukggrp=sumxy/sumx
	drop xy x sumxy sumx

	gen retls_ukggrp=retl_ukggrp-rets_ukggrp

	/* Within index */
	gen w_within=(ret624-ret624_ukggrp)*mcap
	gen xy=ret*w_within if w_within>0
	gen x=w_within if ret~=.&w_within>0
	egen sumxy=sum(xy), by(d)
	egen sumx=sum(x), by(d)
	gen retl_within=sumxy/sumx
	drop xy x sumxy sumx	

	gen xy=ret*w_within if w_within<0
	gen x=w_within if ret~=.&w_within<0
	egen sumxy=sum(xy), by(d)
	egen sumx=sum(x), by(d)
	gen rets_within=sumxy/sumx
	drop w xy x sumxy sumx

	gen retls_within=retl_within-rets_within

	local zeroday = mdy(6,23,2016)

	keep if d>=`zeroday'
	collapse (mean) retls*, by(d)


	foreach v in uk within ukggrp {
		sort d	
		gen index_`v' = 100 if d==`zeroday'
		replace index_`v' = index_`v'[_n-1]*(1+retls_`v'*(d>d[_n-1])) if index_`v'[_n-1]~=.
		gsort -d
		replace index_`v' = index_`v'[_n-1]/(1+retls_`v'[_n-1]*(d<d[_n-1])) if index_`v'[_n-1]~=.
		}

	sort d	
	gen index = 100 if d==`zeroday'
	replace index = index[_n-1]*(1+retls*(d>d[_n-1])) if index[_n-1]~=.
	gsort -d
	replace index = index[_n-1]/(1+retls[_n-1]*(d<d[_n-1])) if index[_n-1]~=.	

	save brexit_indices, replace
	}
	
if `graphs' == 1 {	
	
	use brexit_indices, clear

	keep d index_uk index

	keep if d>=mdy(6,23,2016)
	keep if d<mdy(1,1,2017)


	* S&P and prediction market

	local top = 142
	local bot = 96

	gen rup1=`top' if d>=mdy(6,29,2016)&d<=mdy(6,30,2016)
	gen rdn1=`bot' if d>=mdy(6,29,2016)&d<=mdy(6,30,2016)

	gen rup2=`top' if d>=mdy(7,7,2016)&d<=mdy(7,8,2016)
	gen rdn2=`bot' if d>=mdy(7,7,2016)&d<=mdy(7,8,2016)

	gen rup3=`top' if d>=mdy(7,11,2016)&d<=mdy(7,12,2016)
	gen rdn3=`bot' if d>=mdy(7,11,2016)&d<=mdy(7,12,2016)

	gen rup4=`top' if d>=mdy(10,2,2016)&d<=mdy(10,4,2016)
	gen rdn4=`bot' if d>=mdy(10,2,2016)&d<=mdy(10,4,2016)

	gen rup5=`top' if d>=mdy(11,9,2016)&d<=mdy(11,10,2016)
	gen rdn5=`bot' if d>=mdy(11,9,2016)&d<=mdy(11,10,2016)

	gen rup6=`top' if d>=mdy(12,7,2016)&d<=mdy(12,8,2016)
	gen rdn6=`bot' if d>=mdy(12,7,2016)&d<=mdy(12,8,2016)

	local line1 = `bot'-2
	local line2 = `bot'-4
	local line3 = `bot'-6

	label var rup1 "Johnson exits PM race"
	label var rup2 "Gove exits PM race"
	label var rup3 "May elected leader"
	label var rup4 "May Brexit speech"
	label var rup5 "US election"
	label var rup6 "Commons approves Brexit"

	twoway ///
		(rarea rup1 rdn1 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup2 rdn2 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup3 rdn3 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup4 rdn4 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup5 rdn5 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup6 rdn6 d, sort fcolor(gs13) lcolor(gs13)) ///
		(line index_uk index d, sort color(blue green) lpattern(solid dash)) ///
		, ///
			xlabel( ///
				`=mdy(7,1,2016)' "Jul" ///
				`=mdy(8,1,2016)' "Aug" ///
				`=mdy(9,1,2016)' "Sep" ///
				`=mdy(10,1,2016)' "Oct" ///
				`=mdy(11,1,2016)' "Nov" ///
				`=mdy(12,1,2016)' "Dec" ///		
			) 		///
			ylabel(90 "90" 100 "100" 110 "110" 120 "120" 130 "130" 140 "140", ///
			labcolor(blue) angle(horizontal)) ///
			legend(off) ///
			text(`line1' `=mdy(6,29,2016)' "Johnson exits", placement(1) color(black) size(small)) ///
			text(`line2' `=mdy(7,7,2016)' "Gove exits", placement(1) color(black) size(small)) ///
			text(`line3' `=mdy(7,11,2016)' "May elected leader", placement(1) color(black) size(small)) ///
			text(`line1' `=mdy(10,2,2016)' "May Brexit speech", placement(12) color(black) size(small)) ///
			text(`line2' `=mdy(11,8,2016)' "US election", placement(12) color(black) size(small)) ///
			text(`line3' `=mdy(12,7,2016)' "Commons passes Brexit", placement(12) color(black) size(small)) ///
			text(110 `=mdy(7,15,2016)' "Europe (dashed)", placement(1) color(green) size(small)) ///		
			text(132 `=mdy(7,15,2016)' "UK (solid)", placement(1) color(blue) size(small)) ///	
			ytick(, notick) ///
			ytitle("Index (6/23/16 = 100)", color(blue)) ///
			xtitle(" " "2016") ///
			graphregion(color(white)) bgcolor(white) xsize(10) ysize(7.5) name(fig3, replace)

	graph save "Fig3", replace
	graph export "Fig3.png", replace
	graph export "Fig3.eps", replace

	use brexit_indices, clear

	keep d index_within index_ukggrp index

	keep if d>=mdy(6,23,2016)
	keep if d<mdy(1,1,2017)


	* S&P and prediction market

	local top = 130
	local bot = 96

	gen rup1=`top' if d>=mdy(6,29,2016)&d<=mdy(6,30,2016)
	gen rdn1=`bot' if d>=mdy(6,29,2016)&d<=mdy(6,30,2016)

	gen rup2=`top' if d>=mdy(7,7,2016)&d<=mdy(7,8,2016)
	gen rdn2=`bot' if d>=mdy(7,7,2016)&d<=mdy(7,8,2016)

	gen rup3=`top' if d>=mdy(7,11,2016)&d<=mdy(7,12,2016)
	gen rdn3=`bot' if d>=mdy(7,11,2016)&d<=mdy(7,12,2016)

	gen rup4=`top' if d>=mdy(10,2,2016)&d<=mdy(10,4,2016)
	gen rdn4=`bot' if d>=mdy(10,2,2016)&d<=mdy(10,4,2016)

	gen rup5=`top' if d>=mdy(11,9,2016)&d<=mdy(11,10,2016)
	gen rdn5=`bot' if d>=mdy(11,9,2016)&d<=mdy(11,10,2016)

	gen rup6=`top' if d>=mdy(12,7,2016)&d<=mdy(12,8,2016)
	gen rdn6=`bot' if d>=mdy(12,7,2016)&d<=mdy(12,8,2016)

	local line1 = `bot'-2
	local line2 = `bot'-4
	local line3 = `bot'-6

	label var rup1 "Johnson exits PM race"
	label var rup2 "Gove exits PM race"
	label var rup3 "May elected leader"
	label var rup4 "May Brexit speech"
	label var rup5 "US election"
	label var rup6 "Commons approves Brexit"

	twoway ///
		(rarea rup1 rdn1 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup2 rdn2 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup3 rdn3 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup4 rdn4 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup5 rdn5 d, sort fcolor(gs13) lcolor(gs13)) ///
		(rarea rup6 rdn6 d, sort fcolor(gs13) lcolor(gs13)) ///
		(line index index_within index_ukggrp d, sort color(blue green red) lpattern(solid dash shortdash)) ///
		, ///
			xlabel( ///
				`=mdy(7,1,2016)' "Jul" ///
				`=mdy(8,1,2016)' "Aug" ///
				`=mdy(9,1,2016)' "Sep" ///
				`=mdy(10,1,2016)' "Oct" ///
				`=mdy(11,1,2016)' "Nov" ///
				`=mdy(12,1,2016)' "Dec" ///		
			) 		///
			ylabel(90 "90" 100 "100" 110 "110" 120 "120" 130 "130", ///
			labcolor(blue) angle(horizontal)) ///
			legend(off) ///
			text(`line1' `=mdy(6,29,2016)' "Johnson exits", placement(1) color(black) size(small)) ///
			text(`line2' `=mdy(7,7,2016)' "Gove exits", placement(1) color(black) size(small)) ///
			text(`line3' `=mdy(7,11,2016)' "May elected leader", placement(1) color(black) size(small)) ///
			text(`line1' `=mdy(10,2,2016)' "May Brexit speech", placement(12) color(black) size(small)) ///
			text(`line2' `=mdy(11,8,2016)' "US election", placement(12) color(black) size(small)) ///
			text(`line3' `=mdy(12,7,2016)' "Commons passes Brexit", placement(12) color(black) size(small)) ///
			text(100 `=mdy(10,5,2016)' "Between (dots)", placement(1) color(red) size(small)) ///		
			text(105 `=mdy(7,15,2016)' "Within (dashes)", placement(1) color(green) size(small)) ///		
			text(125 `=mdy(7,15,2016)' "Overall (solid)", placement(1) color(blue) size(small)) ///	
			ytick(, notick) ///
			ytitle("Index (6/23/16 = 100)", color(blue)) ///
			xtitle(" " "2016") ///
			graphregion(color(white)) bgcolor(white) xsize(10) ysize(7.5) name(figA2, replace)

	graph save "FigA2", replace
	graph export "FigA2.png", replace
	graph export "FigA2.eps", replace
	}
