%% Replication of Figure 2 for "Monetary Policy, Bounded Rationality, and Incomplete Markets"
% by Emmanuel Farhi and Iv�n Werning
% 
% Written and prepared by Mikel Petri
% Sunday June 23, 2019

horizons	= [1, 9, 17];	% Horizon for each panel (only 3 panels)
K			= 4;			% Max K in the graph

%% Create and store simulations
param2              = param;
param2.lambda       = 1; % Rigid prices
param2.beta_high    = 0.9945;

rho				= 0.966;
std				= sqrt(0.017);
Nstate			= 21; 

param2			= calibrate_income_process(rho, std, Nstate, param2);
param2.ahigh	= param2.yh*20;
param2.a		= param2.a_NL.^param2.grid_NL*(param2.ahigh - param2.alow) + param2.alow;

liqs	= (exp(linspace(log(1),log(10),10))-1)/(10-1)*(0.09-0.01)+0.01;		% Small liquidity
liqs	= [liqs, (exp(linspace(log(1),log(10),15))-1)/(10-1)*(10-0.1)+0.1];	% High liquidity
liqs	= [liqs, 0.165, 0.230];		% Covering some more space

for j = flip(1:length(liqs))
    file_name = ['./simulations/fig2/simul_L',num2str(liqs(j)*1000,'%05.0f'),'_RF_rigid_T'];
    if exist(fullfile(pwd, [file_name, '01.mat']), 'file') == 0
        param2.delta	= liqs(j)*4*(param2.R-1); % capital share
        param2.gamma	= (2+1.44*4*(param2.R-1))/(1-1.44*4*(param2.R-1))*(1-param2.delta)-param2.delta; % Inverse Frisch to preserve RE response for different delta
		
		fprintf('\n\n====\nLiquidity level = %05.0f \n====\n', liqs(j)*1000);
		econ			= DynamicsReducedForm(param2);
		econ.setfrac_consMC(1e6,300);
		for t=horizons
			if exist(fullfile(cd, [file_name, num2str(t), '.mat']), 'file') == 0
				display(['Calculating T=', num2str(t)]);
				econ.one_period_shock(t,1/1.01,t+20,4);
				save([file_name, num2str(t,'%02.0f'), '.mat'], 'econ', 'param2');
			end
		end
    end
end

%% Load and collect data from simulations
files = dir('./simulations/fig2/simul_L*_RF_rigid_T01.mat')';

y1= zeros(length(files), K, 3);
bcfrac = zeros(length(files), 3);

for i_exp=1:length(horizons)
    for j_file=1:length(files)
        try
            load(['./simulations/fig2/', files(j_file).name(1:end-6), num2str(horizons(i_exp),'%02.0f'), '.mat'], 'econ');
            y1(j_file,:,i_exp) =  (econ.Yt(1,2:(K+1))/econ.Y-1)/(econ.R/econ.Rt(horizons(i_exp))-1);
            bcfrac(j_file,i_exp) = econ.frac_consMC*100;
        catch
            y1(j_file,:,i_exp) = NaN;
            bcfrac(j_file,i_exp) = NaN;
        end
    end
end

bcfrac(bcfrac==-1) = 0;

%% Create plot
figure('position',[.1 .1 1400 350])
co = get(gca,'colororder');
marker = {'o', '^', 's', 'd'};

for i_exp=1:length(horizons)
    subplot(1,3,i_exp)
	plot(0:10:90,ones(10,1),'-','Color', 'k','LineWidth',2);	% RE response
	set(gca, 'ColorOrderIndex', 1);
	hold on;
    for k=1:K
        plot(bcfrac(bcfrac(:,i_exp)>=0), y1(bcfrac(:,i_exp)>=0,k,i_exp), marker{k}, 'MarkerFaceColor', co(k,:));
        ylim([0 1.2]);
        xlim([0 90.1]);
        hold on;
    end
    set(gca,'Color','none');
    set(gca,'XTick',0:10:90)
	grid on
	set(gca, 'XGrid', 'off')
	
	h = legend({'$k=\infty$ (RE)','$$k=1$$','$$k=2$$','$$k=3$$','$$k=4$$'},'Location', 'southwest', 'interpreter', 'latex');
	set(h, 'color', 'w');
	h.AutoUpdate = 'off';
	h.PlotChildren = h.PlotChildren([2:5, 1]);
	
    title(strcat('$\tau=',num2str(horizons(i_exp)-1),'$'), 'Interpreter','Latex', 'FontSize',20);
    ylabel('$\epsilon^k_{\tau}$', 'Interpreter','Latex', 'FontSize',20)
    xlabel('% of borrowing constrained', 'Interpreter','Latex')
    
    hold off;
end

export_fig ('./figures/fig2.pdf', '-transparent');

clearvars -except param